﻿using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using Robotless.Modules.Documenting;
using Robotless.Modules.Utilities.EmitExtensions;

namespace Robotless.Modules.Serializing;

public partial class SerializerGenerator : ISerializerTypeProvider
{
    private readonly ConditionalWeakTable<Type, Type> _cachedSerializers = new();

    private readonly ConditionalWeakTable<Assembly, ModuleBuilder> _cachedModules = new();

    public Type GetSerializerType(Type targetType)
        => _cachedSerializers.GetValue(targetType, GenerateSerializer);

    public IDocumentation? Documentation { get; set; }

    /// <summary>
    /// Create a serializer for the target type.
    /// </summary>
    /// <param name="targetType">Target type to serialize.</param>
    /// <returns>Created serializer type, inherited from <see cref="SnapshotSerializerBase{TTarget}"/>.</returns>
    private Type GenerateSerializer(Type targetType)
    {
        if (!targetType.IsVisible)
            // Emit code cannot access members of private types.
            throw new Exception($"Cannot generate serializer for private type {targetType}.");

        var baseType = typeof(SnapshotSerializerBase<>).MakeGenericType(targetType);
        var module = _cachedModules.GetValue(targetType.Assembly, assembly =>
            DynamicModule.Define($"{assembly.GetName().Name}.Serializing.Serializers"));
        var typeBuilder = module.DefineType("GeneratedSerializer" + targetType,
            TypeAttributes.Public | TypeAttributes.AutoClass | TypeAttributes.BeforeFieldInit,
            baseType);
        var classContext = new ClassContext(targetType, typeBuilder);

        var methodBuilders = new List<ISerializerMethodBuilder>
        {
            new LoaderMethodBuilder(classContext),
            new SaverMethodBuilder(classContext),
            new SchemaMethodBuilder(classContext, Documentation)
        };

        foreach (var member in targetType.GetMembers(
                     BindingFlags.Public | BindingFlags.Instance))
        {
            switch (member)
            {
                case FieldInfo { IsInitOnly: false, IsLiteral: false } fieldInfo:
                    if (member.GetCustomAttribute<SnapshotOptionsAttribute>(true) is { Ignored: true })
                    {
                        continue;
                    }

                    methodBuilders.ForEach(builder => builder.GenerateForField(fieldInfo));
                    break;
                case PropertyInfo
                {
                    CanRead: true, CanWrite: true, SetMethod.IsPublic: true, GetMethod.IsPublic: true
                } propertyInfo:
                    if (member.GetCustomAttribute<SnapshotOptionsAttribute>(true) is { Ignored: true })
                    {
                        continue;
                    }

                    methodBuilders.ForEach(builder => builder.GenerateForProperty(propertyInfo));
                    break;
            }
        }

        methodBuilders.ForEach(builder => builder.Build());

        var type = typeBuilder.CreateType();

        return type;
    }

    private class ClassContext(Type targetType, TypeBuilder typeBuilder)
    {
        public readonly Type TargetType = targetType;

        public readonly Type BaseType = typeof(SnapshotSerializerBase<>).MakeGenericType(targetType);

        public TypeBuilder TypeBuilder { get; } = typeBuilder;

        private readonly Dictionary<Type, FieldBuilder> _serializers = new();

        private FieldBuilder GetSerializerField(Type type)
        {
            if (_serializers.TryGetValue(type, out var field))
                return field;
            field = TypeBuilder.DefineField("Serializer_" + type.Name.Replace('.', '_'),
                typeof(ISnapshotSerializer<>).MakeGenericType(type), FieldAttributes.Public);
            field.SetCustomAttribute(AttributeRequiredMember);
            _serializers[type] = field;
            return field;
        }

        public void EmitLoadSerializer(ILGenerator code, Type targetType)
        {
            if (targetType == TargetType)
            {
                // Use this serializer to serialize itself.
                code.Emit(OpCodes.Ldarg_0);
                return;
            }
            
            // Load the serializer field from the instance.
            code.Emit(OpCodes.Ldarg_0);
            code.Emit(OpCodes.Ldfld, GetSerializerField(targetType));
        }
    }

    private interface ISerializerMethodBuilder
    {
        void Build();

        void GenerateForField(FieldInfo field);

        void GenerateForProperty(PropertyInfo property);
    }

    private static readonly CustomAttributeBuilder AttributeRequiredMember = new(
        typeof(RequiredMemberAttribute).GetConstructor(Type.EmptyTypes)!, []);
}

public static class SerializerGeneratorExtensions
{
    public static TSerializationContext WithGenerator<TSerializationContext>(
        this TSerializationContext context, IDocumentation? documentation = null)
        where TSerializationContext : ISerializationContext
    {
        context.TypeProviders.Add(new SerializerGenerator()
        {
            Documentation = documentation
        });
        return context;
    }
}