﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Serializers.Containers.SharedImplementation;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Containers;

public abstract class ContainerRedirectorBase<TElement, TTarget, TUnderlying, TLoader>(
    ISnapshotSerializer<TElement> elementSerializer)
    : ISnapshotSerializer<TTarget>
    where TUnderlying : TTarget
    where TTarget : IEnumerable<TElement>
    where TLoader : IContainerLoader<TElement,TUnderlying>
{
    public required ISerializationProvider Context { get; init; }

    void ISnapshotSerializer<TTarget>.NewInstance(out TTarget target)
    {
        NewInstance(out var underlying, []);
        target = underlying;
    }

    public void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Array);
        schema.DefineArrayItems(elementSerializer);
    }

    protected abstract void NewInstance(out TUnderlying target, IEnumerable<TElement> elements);

    public void LoadSnapshot(ref TTarget target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.Null)
        {
            target = default!;
            return;
        }
        if (target is not TUnderlying underlying)
            NewInstance(out underlying, target);
        
        TLoader.Load(underlying, reader, elementSerializer);
    }
    
    public void SaveSnapshot(in TTarget? target, IBsonWriter writer)
    {
        if (target == null!)
        {
            writer.WriteNull();
            return;
        }

        EnumerableSaver<TElement>.Save(target, writer, elementSerializer);
    }
}