﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Serializers.Containers.SharedImplementation;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Containers;

/// <summary>
/// This base class provides null value handling,
/// and a private context which can be used to instantiate dependency serializers.
/// </summary>
public abstract class ContainerSerializerBase<TElement, TTarget>
    : ISnapshotSerializer<TTarget> where TTarget : IEnumerable<TElement>
{
    public required ISerializationProvider Context { get; init; }

    public abstract void NewInstance(out TTarget target);
    
    public virtual void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Array);
        schema.DefineArrayItems(ElementSerializer);
    }
    
    protected abstract ISnapshotSerializer<TElement> ElementSerializer { get; }

    public void LoadSnapshot(ref TTarget target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.Null)
        {
            target = default!;
            return;
        }
        if (target == null!)
            NewInstance(out target);
        OnLoadSnapshot(ref target, reader);
    }

    public void SaveSnapshot(in TTarget target, IBsonWriter writer)
    {
        if (target == null!)
        {
            writer.WriteNull();
            return;
        }

        EnumerableSaver<TElement>.Save(target, writer, ElementSerializer);
    }

    protected abstract void OnLoadSnapshot(ref TTarget target, SnapshotReader reader);
}