﻿using Robotless.Modules.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Containers;

public class ContainerSerializerProvider : ISerializerTypeProvider
{
    /// <summary>
    /// Mapping list of interfaces and corresponding serializers,
    /// which can be used to serialize subtypes of these interfaces.
    /// </summary>
    /// <remarks>Only interface with methods to add elements can be added to this list.</remarks>
    private static readonly IReadOnlyList<(Type, Type)> InterfaceMappings = new List<(Type, Type)>()
    {
        (typeof(IList<>), typeof(ListSerializer<>)),
        (typeof(ISet<>), typeof(SetSerializer<>)),
        (typeof(ICollection<>), typeof(CollectionSerializer<>)),
    };

    public Type? GetSerializerType(Type targetType)
    {
        /*
         * ATTENTION: ReadOnly- serializers can only serialize exactly the same target type,
         * for example, ReadOnlyListSerializer<int> can only serialize IReadOnlyList<int>,
         * not some other types which implements IReadOnlyList<int>.
         *
         * This is because those interface-based serializers use interface to serialize containers,
         * but those ReadOnly- interfaces do not provide methods to add elements,
         * and the instantiated underlying containers cannot be assigned to the target type.
         */
        
        var targetDefinition = targetType.IsGenericType? 
            targetType.GetGenericTypeDefinition() : null;
        if (targetDefinition == typeof(KeyValuePair<,>))
        {
            var genericArguments = targetType.GetGenericArguments();
            return genericArguments[0] == typeof(string)
                ? typeof(StringValuePairSerializer<>).MakeGenericType(genericArguments[1])
                : typeof(KeyValuePairSerializer<,>).MakeGenericType(genericArguments);
        }
        
        // Following interfaces are all enumerable.
        if (!targetType.TryMatchInterface(typeof(IEnumerable<>), out _))
            return null;
        
        if (targetDefinition == typeof(IReadOnlyDictionary<,>))
        {
            var genericArguments = targetType.GetGenericArguments();
            return genericArguments[0] == typeof(string)
                ? typeof(ReadOnlyStringDictionarySerializer<>).MakeGenericType(genericArguments[1])
                : typeof(ReadOnlyDictionarySerializer<,>).MakeGenericType(genericArguments);
        }
        // IDictionary<,> serializer without adapter.
        if (targetDefinition == typeof(IDictionary<,>))
        {
            var genericArguments = targetType.GetGenericArguments();
            return genericArguments[0] == typeof(string)
                ? typeof(StringDictionarySerializer<>).MakeGenericType(genericArguments[1])
                : typeof(DictionarySerializer<,>).MakeGenericType(genericArguments);
        }
        
        // IDictionary<,> serializer with adapter.
        if (targetType.TryMatchInterface(typeof(IDictionary<,>), out var targetDictionaryInterface))
        {
            var genericArguments = targetDictionaryInterface.GetGenericArguments();
            if (genericArguments[0] != typeof(string))
                return MakeAdapter(typeof(DictionarySerializer<,>),
                    typeof(IDictionary<,>), 
                    targetDictionaryInterface.GetGenericArguments());
            return typeof(SerializerAdapter<,,>)
                .MakeGenericType(targetType,
                    typeof(IDictionary<,>).MakeGenericType(genericArguments), 
                    typeof(StringDictionarySerializer<>).MakeGenericType(genericArguments[1]));
        }
        
        foreach(var (interfaceType, serializerType) in InterfaceMappings)
        {
            // Those interfaces have been directly registered,
            // so the target type can only be subtypes of them.
            if (targetType.TryMatchInterface(interfaceType, out var targetInterface))
                // Serializer with adapter.
                return MakeAdapter(serializerType, interfaceType, 
                    targetInterface.GetGenericArguments());
        }

        return null;

        Type MakeAdapter(Type serializerDefinition, Type underlyingDefinition, Type[] genericArguments)
        {
            serializerDefinition = serializerDefinition.MakeGenericType(genericArguments);
            underlyingDefinition = underlyingDefinition.MakeGenericType(genericArguments);
            return typeof(SerializerAdapter<,,>)
                .MakeGenericType(targetType, underlyingDefinition, serializerDefinition);
        }
    }
}

public static class ContainerSerializerExtensions
{
    public static TSerializationContext WithContainerSerializers<TSerializationContext>(
        this TSerializationContext context)
        where TSerializationContext : SerializationContext
    {
        // KeyValuePair<,>, IDictionary<,> and IReadOnlyDictionary<,> can not be directly registered,
        // for there are specialized versions for them when keys are strings.
        context.SetSerializer(typeof(IEnumerable<>), typeof(EnumerableSerializer<>));
        context.SetSerializer(typeof(ICollection<>), typeof(CollectionSerializer<>));
        context.SetSerializer(typeof(IReadOnlyCollection<>), typeof(ReadOnlyCollectionSerializer<>));
        context.SetSerializer(typeof(IList<>), typeof(ListSerializer<>));
        context.SetSerializer(typeof(IReadOnlyList<>), typeof(ReadOnlyListSerializer<>));
        context.SetSerializer(typeof(ISet<>), typeof(SetSerializer<>));
        context.SetSerializer(typeof(IReadOnlySet<>), typeof(ReadOnlySetSerializer<>));
        
        context.TypeProviders.Add(new ContainerSerializerProvider());

        return context;
    }
}