﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Serializers.Containers.SharedImplementation;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Containers;

public class DictionarySerializer<TKey, TValue>(
    ISnapshotSerializer<TKey> keySerializer,
    ISnapshotSerializer<TValue> valueSerializer) : ISnapshotSerializer<IDictionary<TKey, TValue>>
    where TKey : notnull
{
    public required ISerializationProvider Context { get; init; }

    public void NewInstance(out IDictionary<TKey, TValue> target)
    {
        target = new Dictionary<TKey, TValue>();
    }
    
    public void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Array);
        using (schema.DefineArrayItemsScope())
        {
            schema.DefineType(SchemaType.Object);
            using (schema.DefinePropertiesScope())
            {
                schema.DefineEmbeddedSchema("!Key", keySerializer);
                schema.DefineEmbeddedSchema("!Value", valueSerializer);
            }
            schema.DefineRequiredProperties("!Key", "!Value");
        }
    }

    public void LoadSnapshot(ref IDictionary<TKey, TValue> target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.EndOfDocument)
            target = null!;
        DictionaryLoader<TKey, TValue>.Load(target, reader, keySerializer, valueSerializer);
    }

    public void SaveSnapshot(in IDictionary<TKey, TValue> target, IBsonWriter writer)
    {
        if (target == null!)
        {
            writer.WriteNull();
            return;
        }
        DictionarySaver<TKey, TValue>.Save(target, writer, keySerializer, valueSerializer);
    }
}