﻿using System.Collections.Frozen;
using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Serializers.Containers.SharedImplementation;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Containers;

public class ReadOnlyDictionarySerializer<TKey, TValue>(
    ISnapshotSerializer<TKey> keySerializer, ISnapshotSerializer<TValue> valueSerializer) : 
    ISnapshotSerializer<IReadOnlyDictionary<TKey, TValue>>
    where TKey : notnull
{
    public required ISerializationProvider Context { get; init; }

    public void NewInstance(out IReadOnlyDictionary<TKey, TValue> target)
    {
        target = new Dictionary<TKey, TValue>();
    }

    public void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Array);
        using (schema.DefineArrayItemsScope())
        {
            schema.DefineType(SchemaType.Object);
            using (schema.DefinePropertiesScope())
            {
                schema.DefineEmbeddedSchema("!Key", keySerializer);
                schema.DefineEmbeddedSchema("!Value", valueSerializer);
            }
            schema.DefineRequiredProperties("!Key", "!Value");
        }
    }

    public void LoadSnapshot(ref IReadOnlyDictionary<TKey, TValue> target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.EndOfDocument)
            target = null!;
        if (target is not IDictionary<TKey, TValue> dictionary)
            dictionary = new Dictionary<TKey, TValue>();
        DictionaryLoader<TKey, TValue>.Load(dictionary, reader, keySerializer, valueSerializer);
    }

    public void SaveSnapshot(in IReadOnlyDictionary<TKey, TValue> target, IBsonWriter writer)
    {
        if (target == null!)
        {
            writer.WriteNull();
            return;
        }
        DictionarySaver<TKey, TValue>.Save(target, writer, keySerializer, valueSerializer);
    }
}