﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Embedded;

public class ArraySerializer<TElement>(ISnapshotSerializer<TElement> elementSerializer) : 
    SnapshotSerializerBase<TElement[]>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Array);
        using (schema.DefineArrayItemsScope())
            elementSerializer.GenerateJsonSchema(schema);
    }

    public override void NewInstance(out TElement[] target)
    {
        target = [];
    }

    protected override void OnLoadSnapshot(ref TElement[] target, SnapshotReader reader)
    {
        reader.ReadStartArray();
        var startingBookmark = reader.GetBookmark();
        var length = 0;
        while (reader.ReadBsonType() != BsonType.EndOfDocument)
        {
            reader.SkipValue();
            ++length;
        }
        
        if (target == null! || target.Length != length)
            target = new TElement[length];
        reader.ReturnToBookmark(startingBookmark);
        for (var index = 0; index < length; ++index)
        {
            elementSerializer.LoadSnapshot(ref target[index], reader);
        }
        reader.ReadEndArray();
    }

    protected override void OnSaveSnapshot(in TElement[] target, IBsonWriter writer)
    {
        writer.WriteStartArray();
        foreach (var element in target)
        {
            elementSerializer.SaveSnapshot(element, writer);
        }
        writer.WriteEndArray();
    }
}