﻿using System.Runtime.CompilerServices;
using System.Runtime.InteropServices.JavaScript;
using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Embedded;

public class MatrixSerializer<TMatrix, TElement>(ISnapshotSerializer<TElement> elementSerializer) :
    SnapshotSerializerBase<TMatrix> where TMatrix : class
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Object);
        using (schema.DefinePropertiesScope())
        {
            schema.WriteName("!Shape");
            using (schema.DocumentScope())
            {
                schema.DefineType(SchemaType.Array);
                schema.WriteName("items");
                using (schema.DocumentScope())
                {
                    schema.DefineType(SchemaType.Integer);
                }
            }
            schema.WriteName("!Values");
            schema.WriteStartDocument();
            
            var rank = typeof(TMatrix).GetArrayRank();
                
            for (var dimension = 0; dimension < rank; ++dimension)
            {
                schema.DefineType(SchemaType.Array);
                schema.WriteName("items");
                schema.WriteStartDocument();
            }
            
            elementSerializer.GenerateJsonSchema(schema);
            
            for (var dimension = 0; dimension < rank; ++dimension)
            {
                schema.WriteEndDocument();
            }
            schema.WriteEndDocument();
        }
    }
    
    public override void NewInstance(out TMatrix target)
    {
        target = null!;
    }

    protected override void OnLoadSnapshot(ref TMatrix target, SnapshotReader reader)
    {
        reader.ReadStartDocument();
        
        var shape = new List<int>();
        
        reader.Locate("!Shape");
        reader.ReadStartArray();
        while (reader.ReadBsonType() != BsonType.EndOfDocument)
        {
            shape.Add(reader.ReadInt32());
        }
        reader.ReadEndArray();
        
        var indices = shape.ToArray();
        var array = Array.CreateInstance(typeof(TElement), indices);
        Array.Clear(indices);
        
        reader.Locate("!Values");
        ReadMatrix(ref reader, array, 0, indices);
        
        reader.ReadEndDocument();
        
        target = Unsafe.As<TMatrix>(array);
    }

    protected override void OnSaveSnapshot(in TMatrix target, IBsonWriter writer)
    {
        var array = Unsafe.As<Array>(target);
        var rank = array.Rank;

        writer.WriteStartDocument();
        
        // Write shape.
        writer.WriteName("!Shape");
        writer.WriteStartArray();
        for (var dimensionIndex = 0; dimensionIndex < rank; ++dimensionIndex)
        {
            writer.WriteInt32(array.GetLength(dimensionIndex));
        }
        writer.WriteEndArray();
        
        // Write value.
        writer.WriteName("!Values");
        WriteMatrix(writer, array, 0, new int[array.Rank]);
        
        writer.WriteEndDocument();
    }
    
    private void WriteMatrix(IBsonWriter writer, Array array, int dimension, int[] indices)
    {
        var length = array.GetLength(dimension);
        
        if (length == 0 && dimension == 0)
        {
            writer.WriteNull();
            return;
        }
        
        writer.WriteStartArray();

        for (var index = 0; index < array.GetLength(dimension); ++index)
        {
            indices[dimension] = index;

            if (dimension == indices.Length - 1)
            {
                var value = (TElement?)array.GetValue(indices);
                if (value == null)
                    writer.WriteNull();
                else
                    elementSerializer.SaveSnapshot(value, writer);
            }
            else
            {
                WriteMatrix(writer, array, dimension + 1, indices);
            }
        }
        
        writer.WriteEndArray();
    }

    private void ReadMatrix(ref SnapshotReader reader, Array array, int dimension, int[] indices)
    {
        reader.ReadStartArray();
        
        for (var index = 0; index < array.GetLength(dimension); ++index)
        {
            indices[dimension] = index;

            if (dimension == indices.Length - 1)
            {
                if (reader.ReadBsonType() == BsonType.Null)
                    continue;
                TElement value = default!;
                elementSerializer.LoadSnapshot(ref value, reader);
                array.SetValue(value, indices);
            }
            else
            {
                ReadMatrix(ref reader, array, dimension + 1, indices);
            }
        }
        
        reader.ReadEndArray();
    }
}