﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Primitives;

/// <remarks>
/// sbyte is serialized as int, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class Integer8Serializer : PrimitiveSerializerBase<sbyte>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref sbyte target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (sbyte)reader.ReadDouble(),
            BsonType.Int32 => (sbyte)reader.ReadInt32(),
            BsonType.Int64 => (sbyte)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(sbyte)}")
        };
    }

    protected override void OnSaveSnapshot(in sbyte target, IBsonWriter writer)
    {
        writer.WriteInt32(target);
    }
}

/// <remarks>
/// byte is serialized as int, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class IntegerUnsigned8Serializer : PrimitiveSerializerBase<byte>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref byte target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (byte)reader.ReadDouble(),
            BsonType.Int32 => (byte)reader.ReadInt32(),
            BsonType.Int64 => (byte)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(byte)}")
        };
    }

    protected override void OnSaveSnapshot(in byte target, IBsonWriter writer)
    {
        writer.WriteInt32(target);
    }
}

/// <remarks>
/// short is serialized as int, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class Integer16Serializer : PrimitiveSerializerBase<short>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref short target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (short)reader.ReadDouble(),
            BsonType.Int32 => (short)reader.ReadInt32(),
            BsonType.Int64 => (short)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(short)}")
        };
    }

    protected override void OnSaveSnapshot(in short target, IBsonWriter writer)
    {
        writer.WriteInt32(target);
    }
}

/// <remarks>
/// ushort is serialized as int, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class IntegerUnsigned16Serializer : PrimitiveSerializerBase<ushort>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref ushort target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (ushort)reader.ReadDouble(),
            BsonType.Int32 => (ushort)reader.ReadInt32(),
            BsonType.Int64 => (ushort)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(ushort)}")
        };
    }

    protected override void OnSaveSnapshot(in ushort target, IBsonWriter writer)
    {
        writer.WriteInt32(target);
    }
}


public class Integer32Serializer : PrimitiveSerializerBase<int>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref int target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (int)reader.ReadDouble(),
            BsonType.Int32 => reader.ReadInt32(),
            BsonType.Int64 => (int)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(int)}")
        };
    }

    protected override void OnSaveSnapshot(in int target, IBsonWriter writer)
    {
        writer.WriteInt32(target);
    }
}

/// <remarks>
/// uint32 is serialized as int64, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class IntegerUnsigned32Serializer : PrimitiveSerializerBase<uint>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref uint target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (uint)reader.ReadDouble(),
            BsonType.Int32 => (uint)reader.ReadInt32(),
            BsonType.Int64 => (uint)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(uint)}")
        };
    }

    protected override void OnSaveSnapshot(in uint target, IBsonWriter writer)
    {
        writer.WriteInt64((int)target);
    }
}

public class Integer64Serializer : PrimitiveSerializerBase<long>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref long target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (long)reader.ReadDouble(),
            BsonType.Int32 => reader.ReadInt32(),
            BsonType.Int64 => reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(long)}")
        };
    }

    protected override void OnSaveSnapshot(in long target, IBsonWriter writer)
    {
        writer.WriteInt64(target);
    }
}

/// <remarks>
/// ulong is serialized as long, which is the same behavior of <see cref="MongoDB.Bson.BsonValue.Create"/>.
/// </remarks>
public class IntegerUnsigned64Serializer : PrimitiveSerializerBase<ulong>
{
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Integer);
    }
    
    protected override void OnLoadSnapshot(ref ulong target, SnapshotReader reader)
    {
        target = reader.CurrentBsonType switch
        {
            BsonType.Double => (ulong)reader.ReadDouble(),
            BsonType.Int32 => (ulong)reader.ReadInt32(),
            BsonType.Int64 => (ulong)reader.ReadInt64(),
            _ => throw new InvalidCastException(
                $"Cannot deserialize {reader.CurrentBsonType} to {typeof(ulong)}")
        };
    }

    protected override void OnSaveSnapshot(in ulong target, IBsonWriter writer)
    {
        writer.WriteInt64((long)target);
    }
}