﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Primitives;

public class ObjectSerializer(ISnapshotSerializer<Type> typeSerializer) :
    PrimitiveSerializerBase<object>
{
    public static SnapshotLayoutVerifier LayoutVerifier { get; } = new(
        new Dictionary<string, Predicate<IBsonReader>>
        {
            ["!Type"] = reader => reader.GetCurrentBsonType() == BsonType.String,
            ["!Value"] = reader => reader.GetCurrentBsonType() is BsonType.Null or BsonType.Document
        });
 
    public override void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Object);
        using (schema.DefinePropertiesScope())
        {
            schema.DefinePrimitiveProperty( 
                "!Type", SchemaType.String, 
                "Assembly qualified type name of the object.");
            schema.DefinePrimitiveProperty(
                "!Value", SchemaType.Object,
                "Snapshot of the object.");
        }
        schema.DefineRequiredProperties("!Type", "!Value");
        schema.DefineNoAdditionalPropertiesAllowed();
    }
    
    protected override void OnLoadSnapshot(ref object target, SnapshotReader reader)
    {
        reader.ReadStartDocument();

        Type type = null!;

        reader.Locate("!Type");
        typeSerializer.LoadSnapshot(ref type, reader);
        if (type == null)
            throw new Exception("Type designated in the snapshot is invalid.");

        reader.Locate("!Value");
        var serializer = Context.GetSerializer(type) ??
                         throw new Exception($"Cannot find serializer for type \"{type}\".");
        
        serializer.LoadSnapshot(ref target, reader);
        
        reader.ReadEndDocument();
    }

    protected override void OnSaveSnapshot(in object target, IBsonWriter writer)
    {
        writer.WriteStartDocument();

        var targetType = target.GetType();

        writer.WriteName("!Type");
        writer.WriteString(targetType.FullName);

        writer.WriteName("!Value");
        var serializer = Context.GetSerializer(targetType) ??
                         throw new Exception($"Cannot find serializer for type \"{targetType}\".");
        serializer.SaveSnapshot(target, writer);

        writer.WriteEndDocument();
    }
}