﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers.Primitives;

/// <summary>
/// Serializers derived from this base class shares these properties: <br/>
/// - Always overwrite the target value with a new instance. <br/>
/// - Accept null values. <br/>
/// - Can be accessed through a static shared instance. <br/>
/// </summary>
public abstract class PrimitiveSerializerBase<TTarget> : ISnapshotSerializer<TTarget>
{
    private static readonly Lazy<ISnapshotSerializer<TTarget>> GlobalInstance = 
        new(() => PrimitiveSerializerContext.Instance.GetSerializer<TTarget>()!, 
            LazyThreadSafetyMode.ExecutionAndPublication);
    
    /// <summary>
    /// Static shared singleton instance of this primitive serializer.
    /// Its context is set to the singleton instance of <see cref="PrimitiveSerializerContext"/>.
    /// </summary>
    public static ISnapshotSerializer<TTarget> Shared { get; } = GlobalInstance.Value;

    public required ISerializationProvider Context { get; init; }

    void ISnapshotSerializer.LoadSnapshot(ref object target, SnapshotReader reader)
    {
        if (target is TTarget castedTarget)
        {
            LoadSnapshot(ref castedTarget, reader);
            target = castedTarget!;
            return;
        }
        TTarget newTarget = default!;
        LoadSnapshot(ref newTarget, reader);
        target = newTarget!;
    }

    public virtual void NewInstance(out TTarget target)
    {
        target = default!;
    }

    public abstract void GenerateJsonSchema(SchemaWriter schema);

    public void LoadSnapshot(ref TTarget target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.Null)
        {
            target = default!;
            return;
        }
        
        OnLoadSnapshot(ref target, reader);
    }

    public void SaveSnapshot(in TTarget target, IBsonWriter writer)
    {
        if (target == null)
        {
            writer.WriteNull();
            return;
        }
        
        OnSaveSnapshot(target, writer);
    }
    
    protected abstract void OnLoadSnapshot(ref TTarget target, SnapshotReader reader);
    
    protected abstract void OnSaveSnapshot(in TTarget target, IBsonWriter writer);
}

/// <summary>
/// Context for static shared instances of primitive serializers.
/// </summary>
public class PrimitiveSerializerContext : SerializationContext
{
    private PrimitiveSerializerContext() {}
    
    public static PrimitiveSerializerContext Instance { get; }

    static PrimitiveSerializerContext()
    {
        Instance = new PrimitiveSerializerContext().WithPrimitiveSerializers();
    }
}