﻿using System.Diagnostics;
using System.Runtime.CompilerServices;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing.Serializers;

/// <summary>
/// Some serializers can serialize objects by only using methods provided by an interface,
/// therefore, they have the ability to serializer all types who implement that interface.
/// But they only implement the snapshot interface for that interface,
/// and they cannot be converted into the snapshot interface for subtypes,
/// which may cause troubles for users. <br/>
/// Intuitively, users expect to get a serializer of <see cref="ISnapshotSerializer{TTarget}"/>> for TTarget,
/// not a serializer for one of its interfaces. <br/>
/// This adapter can wrap the serializer for the interface to a serializer for the target type.
/// </summary>
/// <typeparam name="TTarget">
/// Subtype of the actual target type that specified serializer type can serialize.
/// </typeparam>
/// <typeparam name="TSerializerTarget">Actual target type that specified serializer can serialize.</typeparam>
/// <typeparam name="TSerializer">Serializer type to wrap.</typeparam>
public class SerializerAdapter<TTarget, TSerializerTarget, TSerializer> :
    ISnapshotSerializer<TTarget>
    where TTarget : class, TSerializerTarget
    where TSerializer : ISnapshotSerializer<TSerializerTarget>
{
    public required ISerializationProvider Context { get; init; }

    public required ISnapshotSerializer<TSerializerTarget> Serializer { get; init; }
    
    public void NewInstance(out TTarget target)
    {
        target = Activator.CreateInstance<TTarget>();
    }

    public void GenerateJsonSchema(SchemaWriter schema)
    {
        Serializer.GenerateJsonSchema(schema);
    }

    public void LoadSnapshot(ref TTarget target, SnapshotReader reader)
    {
        TSerializerTarget underlyingTarget = target;
        Serializer.LoadSnapshot(ref underlyingTarget, reader);
        if (underlyingTarget is not TTarget restoredTarget)
            throw new Exception("SerializerAdapter cannot be used with serializers " +
                                "which will replace the target instance with a instance of a different type.");
        target = restoredTarget;
    }

    public void SaveSnapshot(in TTarget target, IBsonWriter writer)
    {
        Serializer.SaveSnapshot(target, writer);
    }
}

public static class SerializerAdapter
{
    /// <summary>
    /// Use <see cref="SerializerAdapter{TTarget, TSerializerTarget, TSerializer}"/> to wrap a serializer type
    /// to add a new interface implementation of the specified target type.
    /// </summary>
    /// <param name="targetType">New target type to add to the specified serializer type.</param>
    /// <param name="serializerType">Serializer type to add a new target type.</param>
    /// <returns>Serializer type with a new interface implementation of the specified new target type.</returns>
    /// <exception cref="Exception">
    /// Throw if the specified serializer type does not implement <see cref="ISnapshotSerializer{T}"/>,
    /// or the target type of the specified serializer type is not assignable to the specified target type.
    /// </exception>
    public static Type Wrap(Type targetType, Type serializerType)
    {
        var actualTargetType = serializerType.GetInterface(typeof(ISnapshotSerializer<>).Name)?
            .GetGenericArguments()[0];
        if (actualTargetType == null)
            throw new Exception("Serializer type must implement ISnapshotSerializer<>");
        if (!targetType.IsAssignableTo(actualTargetType))
            throw new Exception("New target type must be assignable to the target type of the specified serializer type.");
        return typeof(SerializerAdapter<,,>)
            .MakeGenericType(targetType, actualTargetType, serializerType);
    }
}