﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Modules.Serializing;

public abstract class SnapshotSerializerBase<TTarget> : ISnapshotSerializer<TTarget>
{
    public static Type TargetType { get; } = typeof(TTarget);

    public required ISerializationProvider Provider { get; init; }
    
    public bool EnableSubtypeIdentification { get; set; } = 
        !typeof(TTarget).IsValueType && !typeof(TTarget).IsArray;
    
    public virtual void NewInstance(out TTarget target)
    {
        target = Activator.CreateInstance<TTarget>();
    }

    public virtual void GenerateJsonSchema(SchemaWriter schema)
    {
        schema.DefineType(SchemaType.Object);
    }

    public void LoadSnapshot(ref TTarget target, SnapshotReader reader)
    {
        if (reader.GetCurrentBsonType() == BsonType.Null)
        {
            target = default!;
            return;
        }

        // Check if the current value is the snapshot of a derived object.
        if (EnableSubtypeIdentification && ObjectSerializer.LayoutVerifier.IsMatch(reader))
        {
            reader.ReadStartDocument();
            reader.Locate("!Type");
            Type actualType = default!;
            TypeSerializer.Shared.LoadSnapshot(ref actualType, reader);
            var serializer = Provider.RequireSerializer(actualType);
            reader.Locate("!Value");
            object actualValue = default!;
            serializer.LoadSnapshot(ref actualValue, reader);
            target = (TTarget)actualValue;
            reader.ReadEndDocument();
            return;
        }
        
        if (target == null)
            NewInstance(out target);
        OnLoadSnapshot(ref target, reader);
    }

    public void SaveSnapshot(in TTarget target, IBsonWriter writer)
    {
        if (target == null)
        {
            writer.WriteNull();
            return;
        }
        
        // Check if specified object is an instance of derived class of target class.
        if (EnableSubtypeIdentification)
        {
            var actualType = target.GetType();
            if (actualType != TargetType)
            {
                writer.WriteStartDocument();
                writer.WriteName("!Type");
                TypeSerializer.Shared.SaveSnapshot(actualType, writer);
                var serializer = Provider.RequireSerializer(actualType);
                writer.WriteName("!Value");
                serializer.SaveSnapshot(target, writer);
                writer.WriteEndDocument();
                return;
            }
        }
        
        OnSaveSnapshot(target, writer);
    }
    
    protected abstract void OnLoadSnapshot(ref TTarget target, SnapshotReader reader);
    
    protected abstract void OnSaveSnapshot(in TTarget target, IBsonWriter writer);
}