﻿using System.Collections.Concurrent;
using MongoDB.Bson;
using MongoDB.Bson.IO;

namespace Robotless.Modules.Serializing.Utilities;

public class BsonLayoutVerifier(IReadOnlyDictionary<string, Predicate<IBsonReader>> layout)
{
    public IReadOnlyDictionary<string, Predicate<IBsonReader>> Layout { get; } = layout;

    private static readonly ConcurrentBag<HashSet<string>> ReusedSets = new();
    
    public static int MaxReusableCount { get; set; } = Environment.ProcessorCount * 2;
    
    public bool IsMatch(IBsonReader reader)
    {
        var initialPosition = reader.GetBookmark();
    
        if (reader.ReadBsonType() != BsonType.Document)
        {
            reader.ReturnToBookmark(initialPosition);
            return false;
        }
        
        if (!ReusedSets.TryTake(out var requiredFields))
            requiredFields = [];
        requiredFields.UnionWith(Layout.Keys.ToHashSet());
        
        reader.ReadStartDocument();
    
        while (reader.ReadBsonType() != BsonType.EndOfDocument)
        {
            var fieldName = reader.ReadName();
            if (Layout.TryGetValue(fieldName, out var fieldVerifier))
            {
                if (fieldVerifier(reader))
                    requiredFields.Remove(fieldName);
                else 
                    break;
            }
            else reader.SkipValue();
        }
    
        if (ReusedSets.Count < MaxReusableCount)
        {
            requiredFields.Clear();
            ReusedSets.Add(requiredFields);
        }
        
        reader.ReturnToBookmark(initialPosition);
    
        return requiredFields.Count == 0;
    }
}