﻿using MongoDB.Bson;
using MongoDB.Bson.IO;

namespace Robotless.Modules.Serializing.Utilities;

public class BsonValueReader(BsonValue value) : IBsonReader
{
    private BsonDocumentReader? _reader;

    private BsonDocumentReader EnsureReader()
    {
        if (_reader != null)
            return _reader;
        var document = new BsonDocument()
        {
            { "Value", value }
        };
        _reader = new BsonDocumentReader(document);
        _reader.ReadStartDocument();
        _reader.ReadName();
        return _reader;
    }

    public void Dispose()
        => _reader?.Dispose();

    public void Close()
        => _reader?.Close();

    public BsonReaderBookmark GetBookmark()
        => _reader?.GetBookmark() ??
           throw new InvalidOperationException("Current state does not support getting a bookmark.");

    public BsonType GetCurrentBsonType()
        => _reader?.GetCurrentBsonType() ?? value.BsonType;

    public bool IsAtEndOfFile()
        => _reader?.IsAtEndOfFile() ?? false;

    public void PopSettings()
        => _reader?.PopSettings();

    public void PushSettings(Action<BsonReaderSettings> configurator)
        => _reader?.PushSettings(configurator);

    public BsonBinaryData ReadBinaryData()
        => _reader?.ReadBinaryData() ?? value!.AsBsonBinaryData;

    public bool ReadBoolean()
        => _reader?.ReadBoolean() ?? value.AsBoolean;

    public BsonType ReadBsonType()
        => _reader?.ReadBsonType() ?? value.BsonType;

    public byte[] ReadBytes()
        => _reader?.ReadBytes() ?? value.AsBsonBinaryData.Bytes;

    public long ReadDateTime()
        => _reader?.ReadDateTime() ?? value.AsBsonDateTime.MillisecondsSinceEpoch;

    public Decimal128 ReadDecimal128()
        => _reader?.ReadDecimal128() ?? value.AsDecimal128;

    public double ReadDouble()
        => _reader?.ReadDouble() ?? value.AsDouble;

    public void ReadEndArray()
        => _reader?.ReadEndArray();

    public void ReadEndDocument()
        => _reader?.ReadEndDocument();

    public int ReadInt32()
        => _reader?.ReadInt32() ?? value.AsInt32;

    public long ReadInt64()
        => _reader?.ReadInt64() ?? value.AsInt64;

    public string ReadJavaScript()
        => _reader?.ReadJavaScript() ?? value.AsBsonJavaScript.Code;

    public string ReadJavaScriptWithScope()
        => _reader?.ReadJavaScriptWithScope() ?? value.AsBsonJavaScriptWithScope.Code;

    public void ReadMaxKey()
        => _reader?.ReadMaxKey();

    public void ReadMinKey()
        => _reader?.ReadMinKey();

    public string ReadName(INameDecoder nameDecoder)
        => _reader?.ReadName(nameDecoder) ?? 
           throw new InvalidOperationException("Current state does not support reading name.");

    public void ReadNull()
        => _reader?.ReadNull();

    public ObjectId ReadObjectId()
        => _reader?.ReadObjectId() ?? value.AsObjectId;

    public IByteBuffer ReadRawBsonArray()
        => EnsureReader().ReadRawBsonArray();

    public IByteBuffer ReadRawBsonDocument()
        => EnsureReader().ReadRawBsonDocument();

    public BsonRegularExpression ReadRegularExpression()
        => _reader?.ReadRegularExpression() ?? value.AsBsonRegularExpression;

    public void ReadStartArray()
        => EnsureReader().ReadStartArray();

    public void ReadStartDocument()
        => EnsureReader().ReadStartDocument();

    public string ReadString()
        => _reader?.ReadString() ?? value.AsString;

    public string ReadSymbol()
        => _reader?.ReadSymbol() ?? value.AsBsonSymbol.Name;

    public long ReadTimestamp()
        => _reader?.ReadTimestamp() ?? value.AsBsonTimestamp.Value;

    public void ReadUndefined()
        => _reader?.ReadUndefined();

    public void ReturnToBookmark(BsonReaderBookmark bookmark)
        => _reader?.ReturnToBookmark(bookmark);

    public void SkipName()
        => _reader?.SkipName();

    public void SkipValue()
        => _reader?.SkipValue();

    public BsonType CurrentBsonType => _reader?.CurrentBsonType ?? value.BsonType;
    
    public BsonReaderState State => _reader?.State ?? BsonReaderState.Value;
}