﻿using MongoDB.Bson;
using MongoDB.Bson.IO;

namespace Robotless.Modules.Serializing.Utilities;

public class BsonValueWriter : IBsonWriter
{
    private BsonValue _value = BsonNull.Value;

    public BsonValue Value => _document != null ? _document["Value"] : _value;
    
    private BsonDocument? _document = null;
    
    private BsonDocumentWriter? _writer = null;
    
    private BsonDocumentWriter EnsureWriter()
    {
        if (_writer != null)
            return _writer;
        _document = new BsonDocument();
        _writer = new BsonDocumentWriter(_document);
        _writer.WriteStartDocument();
        _writer.WriteName("Value");
        return _writer;
    }

    public void Dispose()
        => _writer?.Dispose();

    public void Close()
        => _writer?.Close();

    public void Flush()
        => _writer?.Flush();

    public void PopElementNameValidator()
        => _writer?.PopElementNameValidator();

    public void PopSettings()
        => _writer?.PopSettings();

    public void PushElementNameValidator(IElementNameValidator validator)
        => _writer?.PushElementNameValidator(validator);

    public void PushSettings(Action<BsonWriterSettings> configurator)
        => _writer?.PushSettings(configurator);

    public void WriteBinaryData(BsonBinaryData binaryData)
    {
        if (_writer != null)
            _writer.WriteBinaryData(binaryData);
        else
            _value = binaryData;
    }

    public void WriteBoolean(bool value)
    {
        if (_writer != null)
            _writer.WriteBoolean(value);
        else
            _value = new BsonBoolean(value);
    }

    public void WriteBytes(byte[] bytes)
    {
        if (_writer != null)
            _writer.WriteBytes(bytes);
        else
            _value = new BsonBinaryData(bytes);
    }

    public void WriteDateTime(long value)
    {
        if (_writer != null)
            _writer.WriteDateTime(value);
        else
            _value = new BsonDateTime(value);
    }

    public void WriteDecimal128(Decimal128 value)
    {
        if (_writer != null)
            _writer.WriteDecimal128(value);
        else
            _value = new BsonDecimal128(value);
    }

    public void WriteDouble(double value)
    {
        if (_writer != null)
            _writer.WriteDouble(value);
        else
            _value = new BsonDouble(value);
    }

    public void WriteEndArray()
    {
        if (_writer != null)
            _writer.WriteEndArray();
        else
            throw new InvalidOperationException("Current state disallows writing an array.");
    }

    public void WriteEndDocument()
    {
        if (_writer != null)
            _writer.WriteEndDocument();
        else
            throw new InvalidOperationException("Current state disallows writing a document.");
    }

    public void WriteInt32(int value)
    {
        if (_writer != null)
            _writer.WriteInt32(value);
        else
            _value = new BsonInt32(value);
    }

    public void WriteInt64(long value)
    {
        if (_writer != null)
            _writer.WriteInt64(value);
        else
            _value = new BsonInt64(value);
    }

    public void WriteJavaScript(string code)
    {
        if (_writer != null)
            _writer.WriteJavaScript(code);
        else
            _value = new BsonJavaScript(code);
    }

    public void WriteJavaScriptWithScope(string code)
    {
        EnsureWriter().WriteJavaScriptWithScope(code);
    }

    public void WriteMaxKey()
    {
        if (_writer != null)
            _writer.WriteMaxKey();
        else
            _value = BsonMaxKey.Value;
    }

    public void WriteMinKey()
    {
        if (_writer != null)
            _writer.WriteMinKey();
        else
            _value = BsonMinKey.Value;
    }

    public void WriteName(string name)
    {
        if (_writer != null)
            _writer.WriteName(name);
        else
            throw new InvalidOperationException("Current state disallows writing a name.");
    }

    public void WriteNull()
    {
        if (_writer != null)
            _writer.WriteNull();
        else
            _value = BsonNull.Value;
    }

    public void WriteObjectId(ObjectId objectId)
    {
        if (_writer != null)
            _writer.WriteObjectId(objectId);
        else
            _value = new BsonObjectId(objectId);
    }

    public void WriteRawBsonArray(IByteBuffer slice)
    {
        EnsureWriter().WriteRawBsonArray(slice);
    }

    public void WriteRawBsonDocument(IByteBuffer slice)
    {
        EnsureWriter().WriteRawBsonDocument(slice);
    }

    public void WriteRegularExpression(BsonRegularExpression regex)
    {
        if (_writer != null)
            _writer.WriteRegularExpression(regex);
        else
            _value = regex;
    }

    public void WriteStartArray()
    {
        EnsureWriter().WriteStartArray();
    }

    public void WriteStartDocument()
    {
        EnsureWriter().WriteStartDocument();
    }

    public void WriteString(string value)
    {
        if (_writer != null)
            _writer.WriteString(value);
        else
            _value = new BsonString(value);
    }

    public void WriteSymbol(string value)
    {
        EnsureWriter().WriteSymbol(value);
    }

    public void WriteTimestamp(long value)
    {
        if (_writer != null)
            _writer.WriteTimestamp(value);
        else
            _value = new BsonTimestamp(value);
    }

    public void WriteUndefined()
    {
        if (_writer != null)
            _writer.WriteUndefined();
        else
            _value = BsonUndefined.Value;
    }

    public long Position => _writer?.Position ?? 0;

    public int SerializationDepth => _writer?.SerializationDepth ?? 1;
    public BsonWriterSettings Settings => _writer?.Settings!;
    public BsonWriterState State => _writer?.State ?? BsonWriterState.Value;
}