﻿using System.ComponentModel;
using MongoDB.Bson;
using MongoDB.Bson.Serialization;
using MongoDB.Bson.Serialization.Attributes;
using MongoDB.Bson.Serialization.Serializers;

namespace Robotless.Modules.Serializing.Utilities;

/// <summary>
/// This class is designed to simplify the creation of JSON object schema documents.
/// </summary>
[BsonSerializer(typeof(SchemaDocumentSerializer))]
public class ObjectSchemaDocument : BsonDocument
{
    public readonly BsonDocument Properties = new();

    private readonly BsonArray _required = new();

    public ObjectSchemaDocument()
    {
        this["type"] = "object";
        this["additionalProperties"] = false;
        this["required"] = _required;
        this["properties"] = Properties;
    }

    public new ObjectSchemaDocument Clone()
    {
        var document = new ObjectSchemaDocument()
        {
            { "type", this["type"].Clone() },
            { "additionalProperties", this["additionalProperties"].Clone() },
            { "required", _required.Clone() },
            { "properties", Properties.Clone() }
        };
        return document;
    }

    public ObjectSchemaDocument(BsonDocument document) : base(document)
    {}

    public void AddProperty(string name, BsonDocument property)
    {
        Properties[name] = property;
        _required.Add(name);
    }
}

[EditorBrowsable(EditorBrowsableState.Never)]
public class SchemaDocumentSerializer() : BsonValueSerializerBase<ObjectSchemaDocument>(BsonType.Document)
{
    protected override ObjectSchemaDocument DeserializeValue(BsonDeserializationContext context,
        BsonDeserializationArgs args)
    {
        args.NominalType = typeof(BsonDocument);
        return new ObjectSchemaDocument(BsonDocumentSerializer.Instance.Deserialize(context, args));
    }

    protected override void SerializeValue(BsonSerializationContext context, BsonSerializationArgs args,
        ObjectSchemaDocument value)
    {
        args.SerializeAsNominalType = true;
        BsonDocumentSerializer.Instance.Serialize(context, args, value);
    }
}

public static class SchemaDocumentExtensions
{
    public static ObjectSchemaDocument WithType(this ObjectSchemaDocument schema, SchemaType type)
    {
        schema["type"] = type.ToSchemaString();
        return schema;
    }

    public static ObjectSchemaDocument WithDescription(this ObjectSchemaDocument schema, string description)
    {
        schema["description"] = description;
        return schema;
    }

    public static ObjectSchemaDocument WithProperty(this ObjectSchemaDocument schema, string name,
        BsonDocument property)
    {
        schema.AddProperty(name, property);
        return schema;
    }
    
    public static ObjectSchemaDocument WithProperty(this ObjectSchemaDocument schema, string name, SchemaType type,
        string? description = null)
    {
        var property = new BsonDocument()
        {
            { "type", type.ToSchemaString() }
        };
        if (description != null)
            property["description"] = description;
        schema.AddProperty(name, property);
        return schema;
    }
}