﻿using System.Diagnostics;
using System.Runtime.CompilerServices;
using MongoDB.Bson;
using MongoDB.Bson.IO;

namespace Robotless.Modules.Serializing.Utilities;

public class SchemaWriter(IBsonWriter writer) : IBsonWriter
{
    /// <summary>
    /// Write a type element into this schema.
    /// </summary>
    /// <param name="type">Schema type to define.</param>
    public void DefineType(SchemaType type)
    {
        writer.WriteString("type", type.ToSchemaString());
    }
    
    /// <summary>
    /// Define an 'enum' element into this schema to strict the possible values.
    /// </summary>
    public BsonWriterExtensions.BsonWriterArrayScope DefineEnum()
    {
        writer.WriteName("enum");
        return new BsonWriterExtensions.BsonWriterArrayScope(writer);
    }

    public const string DefaultField = "default";
    
    public const string ConstField = "const";
    
    /// <summary>
    /// Write a description element into this schema.
    /// </summary>
    /// <param name="description">Description text.</param>
    public void DefineDescription(string description)
    {
        writer.WriteString("description", description);
    }
    
    /// <summary>
    /// Write an 'items' element into this schema.
    /// </summary>
    /// <returns>The schema scope with in 'items' element.</returns>
    public BsonWriterExtensions.BsonWriterDocumentScope DefineArrayItemsScope()
    {
        writer.WriteName("items");
        return new BsonWriterExtensions.BsonWriterDocumentScope(writer);
    }
    
    /// <summary>
    /// Write an 'items' element into this schema,
    /// then use the specified element serializer to write the schema of the array items.
    /// </summary>
    /// <param name="elementSerializer">Element serializer to generate schema.</param>
    public void DefineArrayItems(ISnapshotSerializer elementSerializer)
    {
        writer.WriteName("items");
        writer.WriteStartDocument();
        elementSerializer.GenerateJsonSchema(this);
        writer.WriteEndDocument();
    }
    
    /// <summary>
    /// Write a 'properties' element into this schema.
    /// </summary>
    /// <returns>The scope within the 'properties' element.</returns>
    public BsonWriterExtensions.BsonWriterDocumentScope DefinePropertiesScope()
    {
        writer.WriteName("properties");
        return new BsonWriterExtensions.BsonWriterDocumentScope(writer);
    }
    
    /// <summary>
    /// Write a schema element with the specified name, type, and description.
    /// </summary>
    /// <param name="name">Name of this element.</param>
    /// <param name="type">Schema type of this element.</param>
    /// <param name="description">Description for this element.</param>
    public void DefinePrimitiveProperty(string name, SchemaType type, 
        string? description = null)
    {
        writer.WriteName(name);
        writer.WriteStartDocument();
        DefineType(type);
        if (description != null)
            DefineDescription(description);
        writer.WriteEndDocument();
    }
    
    /// <summary>
    /// Write an array element of 'required' with the specified property names.
    /// This is used to define the required properties of an object schema.
    /// </summary>
    /// <param name="names">Names of required properties.</param>
    public void DefineRequiredProperties(params string[] names)
    {
        writer.WriteName("required");
        writer.WriteStartArray();
        foreach (var name in names)
            writer.WriteString(name);
        writer.WriteEndArray();
    }

    /// <summary>
    /// Write an element stating that no additional properties are allowed.
    /// </summary>
    public void DefineNoAdditionalPropertiesAllowed()
    {
        writer.WriteName("additionalProperties");
        writer.WriteBoolean(false);
    }

    /// <summary>
    /// Write an schema element with the specified name and use the specified serializer to write the schema content.
    /// </summary>
    /// <param name="name">Name of the schema element to write.</param>
    /// <param name="serializer">Serializer to generate the schema content.</param>
    public void DefineEmbeddedSchema(string name, ISnapshotSerializer serializer)
    {
        writer.WriteName(name);
        writer.WriteStartDocument();
        serializer.GenerateJsonSchema(this);
        writer.WriteEndDocument();
    }
    
    #region Redirect to IBsonWriter.

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void Dispose() => writer.Dispose();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void Close() => writer.Close();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void Flush() => writer.Flush();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PopElementNameValidator() 
        => writer.PopElementNameValidator();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PopSettings() => writer.PopSettings();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PushElementNameValidator(IElementNameValidator validator)
        => writer.PushElementNameValidator(validator);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PushSettings(Action<BsonWriterSettings> configurator)
        => writer.PushSettings(configurator);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteBinaryData(BsonBinaryData binaryData)
        => writer.WriteBinaryData(binaryData);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteBoolean(bool value)
        => writer.WriteBoolean(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteBytes(byte[] bytes)
        => writer.WriteBytes(bytes);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteDateTime(long value)
        => writer.WriteDateTime(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteDecimal128(Decimal128 value)
        => writer.WriteDecimal128(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteDouble(double value)
        => writer.WriteDouble(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteEndArray()
        => writer.WriteEndArray();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteInt32(int value)
        => writer.WriteInt32(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteInt64(long value)
        => writer.WriteInt64(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteJavaScript(string code)
        => writer.WriteJavaScript(code);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteJavaScriptWithScope(string code)
        => writer.WriteJavaScriptWithScope(code);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteMaxKey() => writer.WriteMaxKey();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteMinKey() => writer.WriteMinKey();
    
    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteNull() => writer.WriteNull();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteObjectId(ObjectId objectId)
        => writer.WriteObjectId(objectId);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteRawBsonArray(IByteBuffer slice)
        => writer.WriteRawBsonArray(slice);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteRawBsonDocument(IByteBuffer slice)
        => writer.WriteRawBsonDocument(slice);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteRegularExpression(BsonRegularExpression regex)
        => writer.WriteRegularExpression(regex);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteStartArray()
        => writer.WriteStartArray();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteSymbol(string value)
        => writer.WriteSymbol(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteTimestamp(long value)
        => writer.WriteTimestamp(value);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteUndefined()
        => writer.WriteUndefined();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteEndDocument()
        => writer.WriteEndDocument();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteName(string name)
        => writer.WriteName(name);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteStartDocument()
        => writer.WriteStartDocument();
    
    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void WriteString(string value)
        => writer.WriteString(value);

    public long Position
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => writer.Position;
    }

    public int SerializationDepth
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => writer.SerializationDepth;
    }

    public BsonWriterSettings Settings
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => writer.Settings;
    } 
    
    public BsonWriterState State
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => writer.State;
    } 
    
    #endregion
}

public static class SchemaWriterFactory
{
    /// <summary>
    /// Create a schema writer that writes to a BSON document.
    /// </summary>
    /// <param name="document">BSON document to write schema into.</param>
    /// <returns>Created schema writer that writes into the specified BSON document.</returns>
    public static SchemaWriter Create(BsonDocument document)
        => new (new BsonDocumentWriter(document));
    
    /// <summary>
    /// Create a schema writer that writes to a BSON binary stream.
    /// </summary>
    /// <param name="stream">Binary stream to write schema into.</param>
    /// <returns>Created schema writer that writes into the specified binary stream.</returns>
    public static SchemaWriter Create(Stream stream)
        => new (new BsonBinaryWriter(stream));

    /// <summary>
    /// Create a schema writer that writes with a JSON text writer.
    /// </summary>
    /// <param name="textWriter">JSON text writer to use.</param>
    /// <param name="settings">Settings such as whether add the indent or not.</param>
    /// <returns>Created schema writer that writes with the specified JSON text writer.</returns>
    public static SchemaWriter Create(TextWriter textWriter, 
        Action<JsonWriterSettings>? settings = null)
    {
        var jsonWriter = new JsonWriter(textWriter);
        settings?.Invoke(jsonWriter.Settings);
        return new SchemaWriter(jsonWriter);
    }

    /// <summary>
    /// Use the specified serializer to generate a JSON schema and write it into a BSON document.
    /// </summary>
    /// <param name="serializer">Serializer to generate schema with.</param>
    /// <param name="document">Schema document.</param>
    public static void CreateSchema(ISnapshotSerializer serializer, out BsonDocument document)
    {
        document = new BsonDocument();
        var writer = Create(document);
        writer.WriteStartDocument();
        serializer.GenerateJsonSchema(writer);
        writer.WriteEndDocument();
    }
    
    /// <summary>
    /// Use the specified serializer to generate a JSON schema and write it into the binary stream.
    /// </summary>
    /// <param name="serializer">Serializer to generate schema with.</param>
    /// <param name="stream">Stream to write schema into.</param>
    /// <remarks>
    /// The schema will be written as a whole document,
    /// which means it will begin with a "StartOfDocument" and end with a "EndOfDocument".
    /// </remarks>
    public static void CreateSchema(ISnapshotSerializer serializer, Stream stream)
    {
        var writer = Create(stream);
        writer.WriteStartDocument();
        serializer.GenerateJsonSchema(writer);
        writer.WriteEndDocument();
    }
}