﻿using System.Diagnostics;
using System.Runtime.CompilerServices;
using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Utilities;

namespace Robotless.Modules.Serializing.Utilities;

using BsonReaderIndex = Dictionary<string, BsonReaderBookmark>;

/// <summary>
/// This struct helps developers to conveniently write snapshot loader code in
/// for BSON documents which cannot guarantee the order of elements,
/// with the least harm to performance.
/// </summary>
/// <param name="reader">reader for this iterator to use.</param>
public class SnapshotReader(IBsonReader reader) : IDisposable, IBsonReader
{
    private struct ReaderContext(IBsonReader reader) : IDisposable
    {
        private static readonly ObjectPool<BsonReaderIndex> IndexPool = new()
        {
            Creator = () => new BsonReaderIndex(),
            Cleaner = index => index.Clear()
        };
        
        // Cached positions of explored elements, bookmarks point to the beginning of value part.
        private BsonReaderIndex? _index;

        // Furthest explored location to jump back to.
        // It points to the value of the last cached element,
        // or the type of next element of the last matched element.
        private BsonReaderBookmark? _furthestPosition = null;

        // Whether the current position is the furthest explored position.
        private bool _isFurthestPosition  = true;

        public bool TryJumpToFurthestPosition()
        {
            if (_isFurthestPosition)
                return false;
            reader.ReturnToBookmark(_furthestPosition);
            if (reader.State == BsonReaderState.Value) 
                reader.SkipValue();
            _isFurthestPosition = true;
            return true;
        }

        public bool TryJumpToStoredPosition(string name)
        {
            if (_index == null || _index.Count == 0 || !_index.Remove(name, out var bookmark)) 
                return false;
            _furthestPosition ??= reader.GetBookmark();
            _isFurthestPosition = false;
            reader.ReturnToBookmark(bookmark);
            return true;

        }

        public void UpdateFurthestPosition()
        {
            _isFurthestPosition = true;
            _furthestPosition = null;
        }
        
        public void StoreElementPosition(string name, BsonReaderBookmark bookmark)
        {
            _isFurthestPosition = true;
            _furthestPosition = bookmark;
            _index ??= IndexPool.Get();
            _index[name] = bookmark;
        }

        public void Dispose()
        {
            if (_index == null) return;
            IndexPool.Return(_index);
            _index = null;
        }
    }
    
    private readonly Stack<ReaderContext> _storedContexts = new();

    private ReaderContext _context = new(reader);
    
    // /// <summary>
    // /// BSON reader which this index uses.
    // /// </summary>
    // public IBsonReader reader { get; } = reader;
    
    public bool TryLocate(string targetName)
    {
        // Try to locate the target element in the index first.
        if (_context.TryJumpToStoredPosition(targetName))
            return true;
        
        // Return to the furthest explored position if current position is not at that position.
        _context.TryJumpToFurthestPosition();
        
        // Skip the value.
        // The reader may point at the value part if the Locate method has been invoked multiple times in succession.
        if (reader.State == BsonReaderState.Value)
            reader.SkipValue();
        
        // Check if reader has reached the end of the document.
        if (reader.State == BsonReaderState.EndOfDocument ||
            reader.ReadBsonType() == BsonType.EndOfDocument)
            return false;
        
        // Continue exploring the document.
        var currentName = reader.ReadName();
        if (currentName == targetName)
        {
            _context.UpdateFurthestPosition();
            return true;
        }
        
        // Store the current element into the index.
        var bookmark = reader.GetBookmark();
        _context.StoreElementPosition(currentName, bookmark);
        reader.SkipValue();
        
        // Continue exploring the document.
        while (reader.ReadBsonType() != BsonType.EndOfDocument)
        {
            currentName = reader.ReadName();
            if (currentName == targetName)
            {
                _context.UpdateFurthestPosition();
                return true;
            }
            bookmark = reader.GetBookmark();
            _context.StoreElementPosition(currentName, bookmark);
            reader.SkipValue();
        }
        
        return false;
    }
    
    [DebuggerStepThrough, StackTraceHidden]
    public void Locate(string targetName)
    {
        if (!TryLocate(targetName))
            throw new Exception($"Cannot locate required field \"{targetName}\" within the snapshot document.");
    }

    public void Dispose()
    {
        _context.Dispose();
        while (_storedContexts.TryPop(out var context))
            context.Dispose();
    }
    
    public void ReadStartDocument()
    {
        _storedContexts.Push(_context);
        _context = new ReaderContext(reader);
        reader.ReadStartDocument();
    }
    
    public void ReadEndDocument()
    {
        // Make sure an exception will be thrown if there are still remaining elements,
        // and the reader will point to the end if all elements have been read.
        _context.TryJumpToFurthestPosition();
        _context.Dispose();
        reader.ReadEndDocument();
        // Restore the previous context.
        _storedContexts.TryPop(out _context);
    }

    public void SkipToEndOfDocument()
    {
        if (reader.State == BsonReaderState.Name)
            reader.SkipName();
        if (reader.State == BsonReaderState.Value)
            reader.SkipValue();
        while (reader.ReadBsonType() != BsonType.EndOfDocument)
        {
            reader.SkipName();
            reader.SkipValue();
        }
        _context.UpdateFurthestPosition();
    }
    
    #region Redirect IBsonReader methods to reader.

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void Close() => reader.Close();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public BsonReaderBookmark GetBookmark() => reader.GetBookmark();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public BsonType GetCurrentBsonType()
    {
        return reader.State == BsonReaderState.Name ? reader.ReadBsonType() :
            reader.GetCurrentBsonType();
    }

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public bool IsAtEndOfFile() => reader.IsAtEndOfFile();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PopSettings() => reader.PopSettings();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void PushSettings(Action<BsonReaderSettings> configurator) 
        => reader.PushSettings(configurator);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public BsonBinaryData ReadBinaryData() => reader.ReadBinaryData();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public bool ReadBoolean() => reader.ReadBoolean();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public BsonType ReadBsonType() => reader.ReadBsonType();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public byte[] ReadBytes() => reader.ReadBytes();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public long ReadDateTime() => reader.ReadDateTime();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public Decimal128 ReadDecimal128() => reader.ReadDecimal128();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public double ReadDouble() => reader.ReadDouble();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadEndArray() => reader.ReadEndArray();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public int ReadInt32() => reader.ReadInt32();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public long ReadInt64() => reader.ReadInt64();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public string ReadJavaScript() => reader.ReadJavaScript();
    
    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public string ReadJavaScriptWithScope() => reader.ReadJavaScriptWithScope();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadMaxKey() => reader.ReadMaxKey();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadMinKey() => reader.ReadMinKey();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public string ReadName(INameDecoder nameDecoder) => reader.ReadName(nameDecoder);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadNull() => reader.ReadNull();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public ObjectId ReadObjectId() => reader.ReadObjectId();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public IByteBuffer ReadRawBsonArray() => reader.ReadRawBsonArray();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public IByteBuffer ReadRawBsonDocument() => reader.ReadRawBsonDocument();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public BsonRegularExpression ReadRegularExpression() => reader.ReadRegularExpression();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadStartArray() => reader.ReadStartArray();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public string ReadString() => reader.ReadString();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public string ReadSymbol() => reader.ReadSymbol();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public long ReadTimestamp() => reader.ReadTimestamp();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReadUndefined() => reader.ReadUndefined();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void ReturnToBookmark(BsonReaderBookmark bookmark) => reader.ReturnToBookmark(bookmark);

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void SkipName() => reader.SkipName();

    [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
    public void SkipValue() => reader.SkipValue();


    public BsonType CurrentBsonType
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => reader.CurrentBsonType;
    }

    
    public BsonReaderState State
    {
        [DebuggerStepThrough, StackTraceHidden, MethodImpl(MethodImplOptions.AggressiveInlining)]
        get => reader.State;
    } 

    #endregion
}

public static class SnapshotReaderFactory
{
    /// <summary>
    /// Create a snapshot reader from a BSON document.
    /// </summary>
    /// <param name="document">BSON document for the created reader to read.</param>
    /// <returns>Created snapshot reader to read from the specified document.</returns>
    public static SnapshotReader Create(BsonDocument document) 
        => new(new BsonDocumentReader(document));
    
    /// <summary>
    /// Create a snapshot reader from a BSON binary stream.
    /// </summary>
    /// <param name="stream">BSON binary stream to read.</param>
    /// <returns>Created snapshot reader to read from the specified BSON binary stream.</returns>
    public static SnapshotReader Create(Stream stream) 
        => new(new BsonBinaryReader(stream));

    /// <summary>
    /// Create a snapshot reader from a JSON text reader.
    /// </summary>
    /// <param name="textReader">Text reader to read JSON from.</param>
    /// <returns>Created snapshot reader to read with the specified JSON text reader.</returns>
    public static SnapshotReader Create(TextReader textReader)
        => new(new JsonReader(textReader));
}