﻿namespace Robotless.Modules.Utilities;

public class BackgroundLoop(Func<CancellationToken, Task> action)
{
    private (Task Task, CancellationTokenSource Cancellation)? _worker;
    
    private async Task Execute(CancellationToken token)
    {
        try
        {
            while (!token.IsCancellationRequested)
                await action(token);
        }
        catch (OperationCanceledException)
        {}
    }
    
    /// <summary>
    /// Check if the background loop is running.
    /// </summary>
    public bool IsRunning => _worker != null;
    
    /// <summary>
    /// Start the background loop.
    /// It will do nothing if the loop is already running.
    /// </summary>
    public void Start()
    {
        if (_worker != null)
            return;
        var cancellation = new CancellationTokenSource();
        var task = Task.Factory.StartNew(
            async () => await Execute(cancellation.Token), cancellation.Token,
            TaskCreationOptions.LongRunning, TaskScheduler.Default).Unwrap();
        _worker = (task, cancellation);
    }

    /// <summary>
    /// Stop the background loop.
    /// It will do nothing if the loop is not running.
    /// </summary>
    /// <returns>Task which will complete when the background loop stops</returns>
    public async Task Stop()
    {
        var worker = Interlocked.Exchange(ref _worker, null);
        if (worker == null)
            return;
        await worker.Value.Cancellation.CancelAsync();
    }
}