﻿using System.Collections;
using System.Collections.Concurrent;

namespace Robotless.Modules.Utilities;

public class ConcurrentSet<TType> : ICollection<TType>, IReadOnlyCollection<TType> where TType : notnull
{
    public static readonly IReadOnlyCollection<TType> Empty = new ConcurrentSet<TType>();
    
    private readonly ConcurrentDictionary<TType, object?> _dictionary = new();

    public bool Add(TType item)
        => _dictionary.TryAdd(item, null);
    
    public void ExceptWith(IEnumerable<TType> other)
    {
        foreach (var element in other)
            Remove(element);
    }

    public void IntersectWith(IEnumerable<TType> other)
    {
        _dictionary.Clear();
        foreach (var element in other)
            Add(element);
    }

    void ICollection<TType>.Add(TType item)
        => Add(item);
    
    public void Clear()
        => _dictionary.Clear();

    public bool Contains(TType item)
        => _dictionary.ContainsKey(item);

    public void CopyTo(TType[] array, int arrayIndex)
        => _dictionary.Keys.CopyTo(array, arrayIndex);

    public bool Remove(TType item)
        => _dictionary.TryRemove(item, out _);

    public int Count => _dictionary.Count;

    public bool IsEmpty => _dictionary.IsEmpty;
    
    bool ICollection<TType>.IsReadOnly => false;

    public IEnumerator<TType> GetEnumerator()
        => _dictionary.Keys.GetEnumerator();

    IEnumerator IEnumerable.GetEnumerator()
        => GetEnumerator();
}