﻿using System.Collections;
using System.Runtime.CompilerServices;

namespace Robotless.Modules.Utilities;

public class ConditionalWeakEvent<THandler> : IEnumerable<THandler> where THandler : Delegate
{
    private readonly ConditionalWeakTable<object, LinkedList<THandler>> _handlers = new();

    /// <summary>
    /// Add an event handler to the handler list.
    /// </summary>
    /// <param name="owner">Object which owns this event handler.</param>
    /// <param name="handler">Event handler to add.</param>
    public void Add(object owner, THandler handler)
    {
        _handlers.GetOrCreateValue(owner).AddLast(handler);
    }

    /// <summary>
    /// Remove an event handler from the handler list.
    /// </summary>
    /// <param name="owner">Object which owns this event handler.</param>
    /// <param name="handler">Event handler to remove.</param>
    /// <returns>
    /// True if a handler is removed, otherwise false.
    /// </returns>
    public bool Remove(object owner, THandler handler)
    {
        if (_handlers.TryGetValue(owner, out var handlers))
            return handlers.Remove(handler);
        return false;
    }

    /// <summary>
    /// Remove all event handlers owned by the specified object.
    /// </summary>
    /// <param name="owner">Object whose event handlers will be removed.</param>
    /// <returns>
    /// True if any handler of the specified object is removed, otherwise false.
    /// </returns>
    public bool Remove(object owner)
    {
        return _handlers.Remove(owner);
    }

    /// <summary>
    /// Invoke the invokers using the specified functor.
    /// </summary>
    /// <param name="invoker">Functor to invoke the handler.</param>
    public void Invoke(Action<THandler> invoker) => Invoke(invoker, CancellationToken.None);

    /// <summary>
    /// Invoke the invokers using the specified functor.
    /// </summary>
    /// <param name="invoker">Functor to invoke the handler.</param>
    /// <param name="cancellationToken">Cancellation token to cancel the iteration.</param>
    public void Invoke(Action<THandler> invoker, CancellationToken cancellationToken)
    {
        foreach (var (_, handlers) in _handlers)
        {
            if (cancellationToken.IsCancellationRequested)
                break;
            foreach (var handler in handlers.TakeWhile(_ => !cancellationToken.IsCancellationRequested))
                invoker(handler);
        }
    }
    
    /// <summary>
    /// Clear all the handlers in the handler list.
    /// </summary>
    public void Clear()
    {
        _handlers.Clear();
    }

    public IEnumerator<THandler> GetEnumerator()
        => _handlers.SelectMany(pair => pair.Value).GetEnumerator();

    IEnumerator<THandler> IEnumerable<THandler>.GetEnumerator() => GetEnumerator();

    IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();
}