﻿using System.Reflection;
using System.Reflection.Emit;
using System.Runtime.CompilerServices;
using JetBrains.Annotations;

namespace Robotless.Modules.Utilities.EmitExtensions;

public static class EmitLogicExtension
{
    public static void EmitNoOperation(this ILGenerator code)
        => code.Emit(OpCodes.Nop);
    
    public static void If(this ILGenerator code, 
        [InstantHandle] Action<ILGenerator> predicate,
        [InstantHandle] Action<ILGenerator>? whenTrue = null, 
        [InstantHandle] Action<ILGenerator>? whenFalse = null)
    {
        whenTrue ??= EmitNoOperation;
        whenFalse ??= EmitNoOperation;
        
        predicate(code);
        
        var labelFalse = code.DefineLabel();
        var labelEnd = code.DefineLabel();
        
        code.Emit(OpCodes.Brfalse, labelFalse);
        whenTrue.Invoke(code);
        code.Emit(OpCodes.Br, labelEnd);
        code.MarkLabel(labelFalse);
        whenFalse.Invoke(code);
        code.MarkLabel(labelEnd);
    }

    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    public static void IsTrue(this ILGenerator code)
    {
        code.Emit(OpCodes.Ldc_I4_1);
        code.Emit(OpCodes.Ceq);
    }
    
    [MethodImpl(MethodImplOptions.AggressiveInlining)]
    public static void IsFalse(this ILGenerator code)
    {
        code.Emit(OpCodes.Ldc_I4_0);
        code.Emit(OpCodes.Ceq);
    }

    /// <inheritdoc cref="OpCodes.Jmp"/>
    public static void Jump(this ILGenerator code, MethodInfo method)
        => code.Emit(OpCodes.Jmp, method);
    
    /// <inheritdoc cref="OpCodes.Br"/>
    public static void Break(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Br, label);
    
    /// <inheritdoc cref="OpCodes.Brtrue"/>
    public static void BreakIfTrue(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Brtrue, label);
    
    /// <inheritdoc cref="OpCodes.Brfalse"/>
    public static void BreakIfFalse(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Brfalse, label);
    
    /// <inheritdoc cref="OpCodes.Bgt"/>
    public static void BreakIfGreater(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Bgt, label);
    
    /// <inheritdoc cref="OpCodes.Bge"/>
    public static void BreakIfGreaterOrEqual(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Bge, label);
    
    /// <inheritdoc cref="OpCodes.Blt"/>
    public static void BreakIfLess(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Blt, label);
    
    /// <inheritdoc cref="OpCodes.Ble"/>
    public static void BreakIfLessOrEqual(this ILGenerator code, Label label)
        => code.Emit(OpCodes.Ble, label);
}