﻿using System.Collections.Concurrent;
using System.Runtime.CompilerServices;

namespace Robotless.Modules.Utilities;

public delegate void LateInitializer(object instance, LateInitializationScope scope, object? data = null);

/// <summary>
/// This scope will first create uninitialized objects, and initialize them when this scope is disposed.
/// It is designed to solve circular dependencies.
/// </summary>
public class LateInitializationScope : IDisposable
{
    private readonly Stack<(object Instance, LateInitializer Initializer, object? Data)> _objects = [];

    private LateInitializationScope()
    {}
    
    /// <summary>
    /// List of uninitialized objects within this scope.
    /// </summary>
    public IEnumerable<(object Instance, object? Data)> UninitializedObjects
        => _objects.Select(item => (item.Instance, item.Data));

    public int Count => _objects.Count;
    
    /// <summary>
    /// Create an uninitialized instance of the specified type.
    /// </summary>
    /// <param name="type">Type to create uninitialized instance.</param>
    /// <param name="initializer">Delegate to initialize the target object.</param>
    /// <param name="data">Additional data to describe this instance.</param>
    /// <returns>Uninitialized instance.</returns>
    public object New(Type type, LateInitializer initializer, object? data = null)
    {
        if (_isRecycled)
            throw new ObjectDisposedException(nameof(LateInitializationScope));
        var instance = RuntimeHelpers.GetUninitializedObject(type);
        _objects.Push((instance, initializer, data));
        return instance;
    }

    /// <summary>
    /// Create a uninitialized instance of the specified type.
    /// </summary>
    /// <param name="initializer">Delegate to initialize the target object.</param>
    /// <typeparam name="TType">Type to create uninitialized instance.</typeparam>
    /// <returns>Uninitialized instance.</returns>
    public TType New<TType>(LateInitializer initializer)
        => (TType)New(typeof(TType), initializer);

    /// <summary>
    /// Clear the content of this scope.
    /// </summary>
    public void Clear()
    {
        _objects.Clear();
    }
    
    /// <summary>
    /// Merge the initialization list of another scope into this scope.
    /// </summary>
    /// <remarks>The specified scope will be cleared with <see cref="Clear"/> method.</remarks>
    /// <param name="scope">Scope whose objects will be moved into this scope.</param>
    public void Merge(LateInitializationScope scope)
    {
        if (_isRecycled)
            throw new ObjectDisposedException(nameof(LateInitializationScope));
        while (scope._objects.TryPop(out var item))
            _objects.Push(item);
        scope.Clear();
        scope.Dispose();
    }
    
    public void Dispose()
    {
        while (_objects.TryPop(out var item))
        {
            item.Initializer(item.Instance, this);
        }
        _objects.Clear();
        if (_isRecycled)
            return;
        if (Scopes.Count >= MaximumStoredScopes)
            return;
        _isRecycled = true;
        Scopes.Add(this);
    }

    private bool _isRecycled;

    private static readonly ConcurrentBag<LateInitializationScope> Scopes = [];

    public static int MaximumStoredScopes { get; set; } = Environment.ProcessorCount * 2;
    
    public static LateInitializationScope New()
    {
        if (!Scopes.TryTake(out var scope))
            return new LateInitializationScope();
        scope._isRecycled = false;
        return scope;
    }
}