using System.Diagnostics.CodeAnalysis;
using System.Reflection;

namespace Robotless.Modules.Utilities;

public static class MemberInfoUtilities
{
    /// <summary>
    /// Try to get the value of this member if it is <see cref="FieldInfo"/> or <see cref="PropertyInfo"/>.
    /// </summary>
    /// <param name="member">Member to get value from.</param>
    /// <param name="target">Target instance, can be null if the member is a static member.</param>
    /// <param name="value">Value, valid if this method returns true.</param>
    /// <returns>True if the member is a field or property, otherwise false.</returns>
    public static bool TryGetValue(this MemberInfo member, object? target, out object? value)
    {
        switch (member)
        {
            case FieldInfo field:
                value = field.GetValue(target);
                return true;
            case PropertyInfo property:
                value = property.GetValue(target);
                return true;
            default:
                value = null;
                return false;
        }
    }

    /// <summary>
    /// Try to set the value of this member if it is <see cref="FieldInfo"/> or <see cref="PropertyInfo"/>.
    /// </summary>
    /// <param name="member">Member to set value to.</param>
    /// <param name="target">Target instance, can be null if the member is a static member.</param>
    /// <param name="value">Value to set to the member.</param>
    /// <returns>True if the member is a field or property, otherwise false.</returns>
    public static bool TrySetValue(this MemberInfo member, object? target, object? value)
    {
        switch (member)
        {
            case FieldInfo field:
                field.SetValue(target, value);
                return true;
            case PropertyInfo property:
                property.SetValue(target, value);
                return true;
            default:
                return false;
        }
    }

    /// <summary>
    /// Try to get the type of the value of this member if it is <see cref="FieldInfo"/> or <see cref="PropertyInfo"/>.
    /// In specific, <see cref="FieldInfo.FieldType"/> for field, <see cref="PropertyInfo.PropertyType"/> for property.
    /// </summary>
    /// <param name="member">Member to get value type of.</param>
    /// <returns>True if the member is a field or property, otherwise false.</returns>
    public static bool TryGetValueType(this MemberInfo member, [NotNullWhen(true)] out Type? type)
    {
        switch (member)
        {
            case FieldInfo field:
                type = field.FieldType;
                return true;
            case PropertyInfo property:
                type = property.PropertyType;
                return true;
            default:
                type = null;
                return false;
        }
    }
}