﻿using System.Collections.Concurrent;

namespace Robotless.Modules.Utilities;

public class ObjectPool<TObject> where TObject : class
{
    public required Func<TObject> Creator { get; set; }

    public required Action<TObject>? Cleaner { get; set; }

    private readonly ConcurrentBag<TObject> _pool = new();

    public int Capacity { get; set; } = Environment.ProcessorCount;

    public TObject Get()
    {
        if (!_pool.TryTake(out var instance))
            instance = Creator();
        return instance;
    }

    public void Return(TObject instance)
    {
        Cleaner?.Invoke(instance);
        if (_pool.Count < Capacity)
            _pool.Add(instance);
    }

    public void Return(ref TObject? instance)
    {
        var value = Interlocked.Exchange(ref instance, null);
        if (value != null)
            Return(value);
    }
}