﻿using System.Collections.Specialized;
using System.Diagnostics.CodeAnalysis;

namespace Robotless.Modules.Utilities.ObservableContainers;

public class ObservablePriorityQueue<TElement, TPriority> : INotifyCollectionChanged
{
    public event NotifyCollectionChangedEventHandler? CollectionChanged;

    private readonly PriorityQueue<TElement, TPriority> _queue;

    public ObservablePriorityQueue()
    {
        _queue = new();
    }

    public ObservablePriorityQueue(int initialCapacity)
    {
        _queue = new(initialCapacity);
    }
    
    public ObservablePriorityQueue(IComparer<TPriority>? comparer)
    {
        _queue = new(comparer);
    }

    public ObservablePriorityQueue(int initialCapacity, IComparer<TPriority>? comparer)
    {
        _queue = new(initialCapacity, comparer);
    }

    public int Count => _queue.Count;

    public IComparer<TPriority> Comparer => _queue.Comparer;

    public PriorityQueue<TElement, TPriority>.UnorderedItemsCollection UnorderedItems => _queue.UnorderedItems;

    /// <summary>Adds the specified element with associated priority to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</summary>
    /// <param name="element">The element to add to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</param>
    /// <param name="priority">The priority with which to associate the new element.</param>
    public void Enqueue(TElement element, TPriority priority)
    {
        _queue.Enqueue(element, priority);
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, element));
    }

    /// <summary>Returns the minimal element from the <see cref="T:System.Collections.Generic.PriorityQueue`2" /> without removing it.</summary>
    /// <exception cref="T:System.InvalidOperationException">The <see cref="T:System.Collections.Generic.PriorityQueue`2" /> is empty.</exception>
    /// <returns>The minimal element of the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</returns>
    public TElement Peek()
        => _queue.Peek();

    /// <summary>Removes and returns the minimal element from the <see cref="T:System.Collections.Generic.PriorityQueue`2" /> - that is, the element with the lowest priority value.</summary>
    /// <exception cref="T:System.InvalidOperationException">The queue is empty.</exception>
    /// <returns>The minimal element of the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</returns>
    public TElement Dequeue()
    {
        var removedElement = _queue.Dequeue();
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, removedElement));
        return removedElement;
    }

    /// <summary>Removes the minimal element and then immediately adds the specified element with associated priority to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</summary>
    /// <param name="element">The element to add to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</param>
    /// <param name="priority">The priority with which to associate the new element.</param>
    /// <exception cref="T:System.InvalidOperationException">The queue is empty.</exception>
    /// <returns>The minimal element removed before performing the enqueue operation.</returns>
    public TElement DequeueEnqueue(TElement element, TPriority priority)
    {
      var removedElement = _queue.DequeueEnqueue(element, priority);
      CollectionChanged?.Invoke(this,
          new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, removedElement));
      CollectionChanged?.Invoke(this,
          new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, element));
      return removedElement;
    }

    /// <summary>Removes the minimal element from the <see cref="T:System.Collections.Generic.PriorityQueue`2" />, and copies it and its associated priority to the <paramref name="element" /> and <paramref name="priority" /> arguments.</summary>
    /// <param name="element">When this method returns, contains the removed element.</param>
    /// <param name="priority">When this method returns, contains the priority associated with the removed element.</param>
    /// <returns>
    /// <see langword="true" /> if the element is successfully removed; <see langword="false" /> if the <see cref="T:System.Collections.Generic.PriorityQueue`2" /> is empty.</returns>
    public bool TryDequeue([MaybeNullWhen(false)] out TElement element, [MaybeNullWhen(false)] out TPriority priority)
    {
        if (!_queue.TryDequeue(out element, out priority)) 
            return false;
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, element));
        return true;
    }

    /// <summary>Returns a value that indicates whether there is a minimal element in the <see cref="T:System.Collections.Generic.PriorityQueue`2" />, and if one is present, copies it and its associated priority to the <paramref name="element" /> and <paramref name="priority" /> arguments. The element is not removed from the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</summary>
    /// <param name="element">When this method returns, contains the minimal element in the queue.</param>
    /// <param name="priority">When this method returns, contains the priority associated with the minimal element.</param>
    /// <returns>
    /// <see langword="true" /> if there is a minimal element; <see langword="false" /> if the <see cref="T:System.Collections.Generic.PriorityQueue`2" /> is empty.</returns>
    public bool TryPeek([MaybeNullWhen(false)] out TElement element, [MaybeNullWhen(false)] out TPriority priority)
    {
        return _queue.TryPeek(out element, out priority);
    }

    /// <summary>Adds the specified element with associated priority to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />, and immediately removes the minimal element, returning the result.</summary>
    /// <param name="element">The element to add to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</param>
    /// <param name="priority">The priority with which to associate the new element.</param>
    /// <returns>The minimal element removed after the enqueue operation.</returns>
    public TElement EnqueueDequeue(TElement element, TPriority priority)
    {
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, element));
        var removedElement = _queue.EnqueueDequeue(element, priority);
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, removedElement));
        return removedElement;
    }

    /// <summary>Enqueues a sequence of element-priority pairs to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</summary>
    /// <param name="items">The pairs of elements and priorities to add to the queue.</param>
    /// <exception cref="T:System.ArgumentNullException">The specified <paramref name="items" /> argument was <see langword="null" />.</exception>
    public void EnqueueRange(
      IEnumerable<(TElement Element, TPriority Priority)> items)
    {
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, items));
        _queue.EnqueueRange(items);
    }

    /// <summary>Enqueues a sequence of elements pairs to the <see cref="T:System.Collections.Generic.PriorityQueue`2" />, all associated with the specified priority.</summary>
    /// <param name="elements">The elements to add to the queue.</param>
    /// <param name="priority">The priority to associate with the new elements.</param>
    /// <exception cref="T:System.ArgumentNullException">The specified <paramref name="elements" /> argument was <see langword="null" />.</exception>
    public void EnqueueRange(IEnumerable<TElement> elements, TPriority priority)
    {
        CollectionChanged?.Invoke(this,
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, elements));
        _queue.EnqueueRange(elements, priority);
    }

    public bool Remove(
      TElement element,
      [MaybeNullWhen(false)] out TElement removedElement,
      [MaybeNullWhen(false)] out TPriority priority,
      IEqualityComparer<TElement>? equalityComparer = null)
    {
        CollectionChanged?.Invoke(this, 
            new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, element));
        return _queue.Remove(element, out removedElement, out priority, equalityComparer);
    }

    /// <summary>Removes all items from the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</summary>
    public void Clear()
    {
        CollectionChanged?.Invoke(
            this, new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Remove, 
                _queue.UnorderedItems));
        _queue.Clear();
    }

    /// <summary>Ensures that the <see cref="T:System.Collections.Generic.PriorityQueue`2" /> can hold up to <paramref name="capacity" /> items without further expansion of its backing storage.</summary>
    /// <param name="capacity">The minimum capacity to be used.</param>
    /// <exception cref="T:System.ArgumentOutOfRangeException">The specified <paramref name="capacity" /> is negative.</exception>
    /// <returns>The current capacity of the <see cref="T:System.Collections.Generic.PriorityQueue`2" />.</returns>
    public int EnsureCapacity(int capacity)
        => _queue.EnsureCapacity(capacity);

    /// <summary>Sets the capacity to the actual number of items in the <see cref="T:System.Collections.Generic.PriorityQueue`2" />, if that is less than 90 percent of current capacity.</summary>
    public void TrimExcess()
        => _queue.TrimExcess();
}