namespace Robotless.Modules.Utilities.Referencing;

public interface IRefCollection<TValue> : IReadOnlyRefCollection<TValue>
{
    /// <summary>
    /// Add a value into this collection and return a reference to it.
    /// </summary>
    /// <param name="value">Optional initial value.</param>
    /// <returns>Reference to this newly added value.</returns>
    IReference<TValue> Add(TValue value = default!);

    /// <summary>
    /// Remove a value by its reference.
    /// </summary>
    /// <param name="reference">Reference to the value to remove.</param>
    /// <returns>True if the value is removed, false if the reference is invalid.</returns>
    bool Remove(IReference<TValue> reference);

    /// <summary>
    /// Try to remove a value from this collection.
    /// This first value equal to the specified value will be removed.
    /// </summary>
    /// <param name="value">Value to search and remove.</param>
    /// <returns>
    /// True if the value is removed,
    /// false if no value in the collection matches the provided value.
    /// </returns>
    bool Remove(TValue value);

    /// <summary>
    /// Clear all the value in this collection.
    /// </summary>
    void Clear();
    
    /// <summary>
    /// Check if this collection contains the specified value.
    /// </summary>
    /// <param name="value">Value to check.</param>
    /// <returns>True if there is a value equal to the specified value in this list.</returns>
    bool Contains(TValue value);
    
    /// <summary>
    /// Check if this collection contains the specified reference, and if this reference is still valid.
    /// </summary>
    /// <param name="reference">Reference to check.</param>
    /// <returns>True if this reference belongs to this collection and it is still valid.</returns>
    bool Contains(IReference<TValue> reference);
    
    /// <summary>
    /// Try to get the reference according to the address of a value in this collection.
    /// </summary>
    /// <param name="value">The value by reference.</param>
    /// <returns>
    /// The corresponding reference,
    /// or null if the address of the specified value does not belong to this collection.
    /// </returns>
    IReference<TValue>? GetReference(ref TValue value);
}