﻿using System.Collections;

namespace Robotless.Modules.Utilities.Referencing;

public partial class RefCollection<TValue> : IReadOnlyRefCollection<TValue>
{
    public struct RefCollectionEnumerator : IRefEnumerator<TValue>
    {
        private RefCollection<TValue>? _collection;

        private int _version;

        private int _tailPosition;
        
        private int _currentPosition;
        
        public RefCollectionEnumerator(RefCollection<TValue> collection)
        {
            _collection = collection;
            _version = collection._operationVersion;
            _currentPosition = -1;
            _tailPosition = _collection.Count + _collection._slots.Count;
        }

        public ref TValue Current
        {
            get
            {
                if (_collection == null)
                    throw new ObjectDisposedException(nameof(RefCollectionEnumerator));
                if (_collection._operationVersion != _version)
                    throw new InvalidOperationException("The group has been modified during the enumeration.");
                return ref _collection!._elements[_currentPosition].Value;
            }
        }
        
        public bool MoveNext()
        {
            if (_collection == null)
                throw new ObjectDisposedException(nameof(RefCollectionEnumerator));
            if (_collection._operationVersion != _version)
                throw new InvalidOperationException("This collection has been modified during the enumeration.");
            
            while (_currentPosition < _tailPosition && !_collection._elements[_currentPosition].IsValid)
                ++_currentPosition;

            return _currentPosition < _tailPosition;
        }

        public void Reset()
        {
            if (_collection == null)
                throw new ObjectDisposedException(nameof(RefCollectionEnumerator));
            _version = _collection._operationVersion;
            _currentPosition = -1;
            _tailPosition = _collection.Count + _collection._slots.Count;
        }
        
        public void Dispose()
        {
            _collection = null;
        }
        
        object IEnumerator.Current => Current!;
        
        TValue IEnumerator<TValue>.Current => Current;
    }

    IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();

    IEnumerator<TValue> IEnumerable<TValue>.GetEnumerator() => GetEnumerator();

    IRefEnumerator<TValue> IRefEnumerable<TValue>.GetEnumerator() => GetEnumerator();
    
    public RefCollectionEnumerator GetEnumerator() => new(this);
}