﻿using System.Runtime.CompilerServices;

namespace Robotless.Modules.Utilities.Referencing;

public partial class RefCollection<TValue> : IRefCollection<TValue>
{
    public static readonly IReadOnlyRefCollection<TValue> Empty = new RefCollection<TValue>();
    
    private struct Element(TValue value)
    {
        public TValue Value = value;

        /// <summary>
        /// Verify this element is valid by checking its reference.
        /// </summary>
        public bool IsValid => Reference != null;
        
        /// <summary>
        /// Reference associated with the current version of this element. <br/>
        /// Not null when this element is in use, otherwise false.
        /// </summary>
        public RefCollectionReference? Reference = null;
    }

    private Element[] _elements = [];

    /// <summary>
    /// Version of the <see cref="_elements"/> array.
    /// </summary>
    private int _elementsVersion;
    
    /// <summary>
    /// It increases when <see cref="Add"/>, <see cref="Remove(IReference{TValue})"/>,
    /// <see cref="Remove(TValue)"/> or <see cref="Clear"/> is invoked.
    /// This field is used to halt enumeration when this collection is modified.
    /// </summary>
    private int _operationVersion;

    /// Count of valid elements.
    public int Count { get; private set; }

    /// Indices of invalid elements, which can be reused.
    private readonly LinkedList<int> _slots = [];

    private readonly Lock _lock = new();
    
    /// <summary>
    /// Clear all content in this group.
    /// </summary>
    public void Clear()
    {
        using var locking = _lock.EnterScope();
        ++_elementsVersion;
        _elements = [];
        ++_operationVersion;
    }

    public IReference<TValue> Add(TValue value)
    {
        using var locking = _lock.EnterScope();
        
        var selectedIndex = Count;

        if (_slots.First != null)
        {
            selectedIndex = _slots.First.Value;
            _slots.RemoveFirst();
        } 
        else if (selectedIndex >= _elements.Length)
        {
            var newCapacity = _elements.Length == 0 ? 4 : 2 * _elements.Length;
            // 2147483591 is the lenght limitation of .NET runtime.
            if ((uint) newCapacity > 2147483591U)
                newCapacity = 2147483591;
            var newElements = new Element[newCapacity];
            // Count of used positions = count of valid elements + count of invalid elements.
            Array.Copy(_elements, newElements, Count + _slots.Count);
            _elements = newElements;
        }

        ref var selectedElement = ref _elements[selectedIndex];
        selectedElement = new Element(value);
        
        ++Count;

        var reference = new RefCollectionReference(this, _elementsVersion, selectedIndex);
        selectedElement.Reference = reference;
        
        ++_operationVersion;
        return reference;
    }

    public bool Remove(IReference<TValue> reference)
    {
        using var locking = _lock.EnterScope();
        var index = RetrieveIndex(reference);
        return index >= 0 && RemoveElement(index);
    }

    public bool Remove(TValue value)
    {
        using var locking = _lock.EnterScope();
        
        var target = -1;

        for (var index = 0; index < _elements.Length; ++index)
        {
            var element = _elements[index];
            if (!element.IsValid)
                continue;
            if (!value!.Equals(element.Value))
                continue;
            target = index;
            break;
        }
        if (target < 0)
            return false;
        RemoveElement(target);
        return true;
    }

    public bool Contains(TValue value)
    {
        using var locking = _lock.EnterScope();
        return _elements.Where(element => element.IsValid)
            .Any(element => value!.Equals(element.Value));
    }
    
    public bool Contains(IReference<TValue> reference)
    {
        using var locking = _lock.EnterScope();
        return RetrieveIndex(reference) >= 0;
    }
    
    public IReference<TValue>? GetReference(ref TValue value)
    {
        if (_elements.Length == 0)
            return null;
        unsafe
        {
            _lock.Enter();
            using var locking = _lock.EnterScope();
            var firstElementAddress = new IntPtr(Unsafe.AsPointer(ref _elements[0].Value));
            var lastElementAddress = new IntPtr(Unsafe.AsPointer(ref _elements[^1].Value));
            var targetPointer = new IntPtr(Unsafe.AsPointer(ref value));
            if (targetPointer < firstElementAddress || targetPointer > lastElementAddress)
                return null;
            var index = (targetPointer - firstElementAddress).ToInt64() / Unsafe.SizeOf<Element>();
            return _elements[index].Reference;
        }
    }

    private bool RemoveElement(int index)
    {
        if (index < 0 || index > _elements.Length)
            return false;
        
        ref var element = ref _elements[index];
        
        if (!element.IsValid)
            return false;
        
        --Count;
        element.Value = default!;
        element.Reference = null;

        var position = _slots.First;
        // Make sure the position is added in ascending order.
        while (position != null && position.Value < index)
        {
            position = position.Next;
        }
        if (position != null)
            _slots.AddAfter(position, new LinkedListNode<int>(index));
        else
            _slots.AddLast(new LinkedListNode<int>(index));
        
        ++_operationVersion;
        
        return true;
    }
    
    private int RetrieveIndex(IReference<TValue> reference)
    {
        if (reference is not RefCollectionReference pointer)
            return -1;
        var (container, version, index) = pointer;
        if (container != this || version != _elementsVersion || 
            index >= _elements.Length || _elements[index].Reference != reference)
            return -1;
        return index;
    }
}