﻿using System.Collections.Concurrent;

namespace Robotless.Modules.Utilities;

public static class SharedObjectPool<TObject> where TObject : class
{
    private static readonly ConcurrentBag<TObject> Bag = new();

    public static int Count => Bag.Count;

    /// <summary>
    /// Maximum capacity for storing objects.
    /// If the count exceeds this capacity, instances returned to the pool will not be stored.
    /// </summary>
    public static int Capacity { get; set; } = Environment.ProcessorCount * 2;
    
    public static TObject? Take() => Bag.TryTake(out var instance) ? instance : null;

    public static bool Return(TObject instance)
    {
        if (Count >= Capacity) 
            return false;
        Bag.Add(instance);
        return true;
    }
    
    public static void Clear() => Bag.Clear();

    public static SharedObjectScope<TObject> Scope(out TObject instance, Func<TObject>? factory = null)
    {
        instance = Take() ?? factory?.Invoke() ?? Activator.CreateInstance<TObject>() ??
                throw new Exception($"Failed to instantiate a new instance of #{typeof(TObject)}.");
        return new SharedObjectScope<TObject>(instance);
    }
}

public ref struct SharedObjectScope<TObject>(TObject value) : IDisposable
    where TObject : class
{
    private TObject? _value = value;
    
    public void Dispose()
    {
        if (_value == null) 
            return;
        SharedObjectPool<TObject>.Return(_value);
        _value = null!;
    }
    
    public static implicit operator TObject(SharedObjectScope<TObject> rent) => rent._value ?? 
        throw new InvalidOperationException("Attempt to access a shared object scope which has been disposed.");
}