﻿using System.Diagnostics.CodeAnalysis;
using System.Reflection;

namespace Robotless.Modules.Utilities;

public static class TypeExtensions
{
    public static bool TryMatchInterface(this Type type, Type interfaceType, 
        [NotNullWhen(true)]out Type? matchedInterfaceType)
    {
        var isInterfaceGenericDefinition = interfaceType.IsGenericTypeDefinition;
        
        if (type == interfaceType || isInterfaceGenericDefinition && type.IsGenericType &&
            type.GetGenericTypeDefinition() == interfaceType)
        {
            matchedInterfaceType = type;
            return true;
        }
        // Here, matched interface type is probably a interface with the same name of the specified generic interface,
        // this may happen when there is a generic interface and a non-generic interface with the same name.
        matchedInterfaceType = type.GetInterface(interfaceType.Name);
        if (matchedInterfaceType == null)
            return false;
        // This also suits the situation when the provided interface is a specialized generic interface.
        if (matchedInterfaceType == interfaceType)
            return true;
        // If matched interface is generic and target is generic definition,
        // or matched interface is not generic and target is not generic definition,
        if (matchedInterfaceType.IsGenericType == isInterfaceGenericDefinition)
            return true;
        
        // Now the matchInterfaceType is matching a non-generic interface which has the same name
        // as that of the specified generic interface.
        foreach (var candidateInterface in type.GetInterfaces())
        {
            if (isInterfaceGenericDefinition)
            {
                if (!candidateInterface.IsGenericType)
                    continue;
                if (candidateInterface.GetGenericTypeDefinition() != interfaceType)
                    continue;
                matchedInterfaceType = candidateInterface;
                return true;
            }

            if (candidateInterface != interfaceType)
                continue;
            matchedInterfaceType = candidateInterface;
            return true;
        }

        matchedInterfaceType = null;
        return false;
    }
    
    private static TObject GenericGetDefaultValue<TObject>()
    {
        return default!;
    }

    /// <summary>
    /// Get the default value for the specified type. This method is equal to `default(T)`.
    /// </summary>
    /// <param name="type">Type to get default value for.</param>
    /// <returns>Boxed default value for the specified type.</returns>
    public static object? GetDefaultValue(this Type type)
    {
        return typeof(TypeExtensions).GetMethod(nameof(GenericGetDefaultValue), 
                BindingFlags.NonPublic | BindingFlags.Static)!
            .MakeGenericMethod(type).Invoke(null, null);
    }
}