﻿using System.Collections.Concurrent;

namespace Robotless.Modules.Utilities;

public class WeakPool<TObject> where TObject: class
{
    private readonly ConcurrentBag<WeakReference<TObject>> _objects = [];

    /// <summary>
    /// Count of objects stored in this pool.
    /// </summary>
    public int Count => _objects.Count;
    
    /// <summary>
    /// This functor will be used to instantiate a new instance when there is no one to use in the pool.
    /// </summary>
    public required Func<TObject> Constructor { get; set; }
    
    /// <summary>
    /// Recycle a instance into this pool.
    /// </summary>
    /// <param name="instance">Instance to store.</param>
    public void Recycle(TObject instance)
    {
        _objects.Add(new WeakReference<TObject>(instance));
    }

    /// <summary>
    /// Take an instance from this pool.
    /// </summary>
    /// <returns>Object instance.</returns>
    public TObject Get()
    {
        TObject? instance = null;
        while (_objects.TryTake(out var reference) && !reference.TryGetTarget(out instance)) 
        {}
        instance ??= Constructor();
        return instance!;
    }

    /// <summary>
    /// Clear all objects in this pool.
    /// </summary>
    public void Clear() => _objects.Clear();
}