using Robotless.Framework;
using Robotless.Kernel;
using Robotless.Modules.Developing;
using Robotless.Modules.Injecting;

namespace Robotless.Platform.Test.Modules.Framework;

[TestFixture, TestOf(typeof(Entity))]
public class TestEntity
{
    private class StubEntity : Entity
    {}

    private class StubComponent : Component
    {
        public int AttachInvokeCount { get; private set; }
        
        public int DetachInvokeCount { get; private set; }
        
        protected override void OnAttach()
        {
            AttachInvokeCount++;
        }

        protected override void OnDetach()
        {
            DetachInvokeCount++;
        }
    }

    private class StubComponentWithDependency : Component
    {
        [ComponentDependency] public StubComponent? Dependency { get; init; }
    }

    private class StubComponentWithEnsuringDependency : Component
    {
        [ComponentDependency(IsEnsuring = true)] public StubComponent? Dependency { get; init; }
    }
    
    [Test]
    public void Entity_AddComponent_NotNull()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };

        var stubComponent = stubEntity.AttachComponent<StubComponent>();
        
        Assert.That(stubComponent, Is.Not.Null);
    }
    
    [Test]
    public void Entity_AddComponent_OnAttach()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };
        
        var stubComponent = stubEntity.AttachComponent<StubComponent>();
        
        Assert.That(stubComponent.AttachInvokeCount, Is.EqualTo(1));
    }
    
    [Test]
    public void Entity_AddComponent_OnDetach()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };
        
        var stubComponent = stubEntity.AttachComponent<StubComponent>();
        stubEntity.DetachComponent<StubComponent>();
        
        Assert.That(stubComponent.DetachInvokeCount, Is.EqualTo(1));
    }

    [Test]
    public void Entity_ComponentInjection_NonEnsuring_NotSatisified()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };
        
        var stubComponent = stubEntity.AttachComponent<StubComponentWithDependency>();
        
        Assert.That(stubComponent.Dependency, Is.Null);
    }
    
    [Test]
    public void Entity_ComponentInjection_NonEnsuring_Satisfied()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };

        var dependency = stubEntity.AttachComponent<StubComponent>();
        var dependent = stubEntity.AttachComponent<StubComponentWithDependency>();
        
        Assert.That(dependent.Dependency, Is.EqualTo(dependency));
    }
    
    [Test]
    public void Entity_ComponentInjection_Ensuring()
    {
        var workspace = new DevelopmentWorkspace();
        
        var stubEntity = new StubEntity()
        {
            Workspace = workspace
        };
        
        var dependent = stubEntity.AttachComponent<StubComponentWithEnsuringDependency>();
        var dependency = stubEntity.GetComponent<StubComponent>();
        Assert.Multiple(() =>
        {
            Assert.That(dependency, Is.Not.Null);
            Assert.That(dependent.Dependency, Is.EqualTo(dependency));
        });
    }
}