﻿using System.Diagnostics.CodeAnalysis;
using Robotless.Modules.Injecting;
using Robotless.Modules.Injecting.Injectors;

namespace Robotless.Platform.Test.Modules.Injecting;

[TestFixture, TestOf(typeof(ConstructorInjector))]
public class TestConstructorInjector
{
    [Test]
    public void Create_Injector_NotNull()
    {
        var injector = ConstructorInjector.Get(typeof(StubOneConstructorInjectionClass));
        Assert.That(injector, Is.Not.Null);
    }

    private class StubOneConstructorInjectionClass(string text, int number, double value)
    {
        public readonly string Text = text;
    
        public readonly int Number = number;
    
        public readonly double Value = value;
    }

    
    [Test]
    public void Create_Class_OneConstructor()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5)
            .AddConstant("Sample");
        var succeeded = ConstructorInjector.TryInject<StubOneConstructorInjectionClass>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target, Is.Not.Null);
        });
        Assert.Multiple(() =>
        {
            Assert.That(target.Text, Is.EqualTo("Sample"));
            Assert.That(target.Number, Is.EqualTo(1));
            Assert.That(target.Value, Is.EqualTo(0.5));
        });
    }

    private class StubTwoConstructorInjectionClass(string text, int number, int value)
    {
        public StubTwoConstructorInjectionClass(int number, int value1, int value2, int value3) : 
            this(string.Empty, number, value1 + value2 + value3)
        {}
    
        public readonly string Text = text;
    
        public readonly int Number = number;
    
        public readonly double Value = value;
    }
    
    [Test]
    public void Create_Class_TwoConstructor()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5);
        var succeeded = ConstructorInjector.TryInject<StubTwoConstructorInjectionClass>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target, Is.Not.Null);
        });
        Assert.Multiple(() =>
        {
            Assert.That(target.Text, Is.EqualTo(string.Empty));
            Assert.That(target.Number, Is.EqualTo(1));
            Assert.That(target.Value, Is.EqualTo(3));
        });
    }

    [Test]
    public void Create_Class_Unsatisfied_Null()
    {
        var container = new InjectionContainer()
            .AddConstant(0.5);
        var succeeded = ConstructorInjector.TryInject<StubOneConstructorInjectionClass>(
            out var target, container, out var missing);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.False);
            Assert.That(target, Is.Null);
        });
    }

    private struct StubOneConstructorInjectionStruct(string text, int number, double value)
    {
        public readonly string Text = text;
    
        public readonly int Number = number;
    
        public readonly double Value = value;
    }

    
    [Test]
    public void Create_Struct_OneConstructor()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5)
            .AddConstant("Sample");
        var succeeded = ConstructorInjector.TryInject<StubOneConstructorInjectionStruct>(
            out var target, container, out _);
        Assert.That(succeeded, Is.True);
        Assert.Multiple(() =>
        {
            Assert.That(target.Text, Is.EqualTo("Sample"));
            Assert.That(target.Number, Is.EqualTo(1));
            Assert.That(target.Value, Is.EqualTo(0.5));
        });
    }

    private struct StubTwoConstructorInjectionStruct(string text, int number, int value)
    {
        public StubTwoConstructorInjectionStruct(int number, int value1, int value2, int value3) : 
            this(string.Empty, number, value1 + value2 + value3)
        {}
    
        public readonly string Text = text;
    
        public readonly int Number = number;
    
        public readonly double Value = value;
    }
    
    [Test]
    public void Create_Struct_TwoConstructor()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5);
        var succeeded = ConstructorInjector.TryInject<StubTwoConstructorInjectionStruct>(
            out var target, container, out _);
        Assert.That(succeeded, Is.True);
        Assert.Multiple(() =>
        {
            Assert.That(target.Text, Is.EqualTo(string.Empty));
            Assert.That(target.Number, Is.EqualTo(1));
            Assert.That(target.Value, Is.EqualTo(3));
        });
    }

    [Test]
    public void Create_Struct_Unsatisfied_Null()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5);
        var succeeded = ConstructorInjector.TryInject<StubOneConstructorInjectionStruct>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.False);
            Assert.That(target, Is.EqualTo(default(StubOneConstructorInjectionStruct)));
        });
    }

    [DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.PublicFields | 
                                DynamicallyAccessedMemberTypes.PublicProperties)]
    private class StubWithDefaultArguments(int arg1, double arg2 = double.Pi)
    {
        public int Integer = arg1;

        public double Floating = arg2;
    }

    [Test]
    public void Injector_WithDefaultArguments()
    {
        var container = new InjectionContainer()
            .AddConstant(1);
        var succeeded = ConstructorInjector.TryInject<StubWithDefaultArguments>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target?.Integer, Is.EqualTo(1));
            Assert.That(target?.Floating, Is.EqualTo(double.Pi));
        });
    }

    [Test]
    public void Injector_WithDefaultArguments_ShouldFail()
    {
        var container = new InjectionContainer();
        var succeeded = ConstructorInjector.TryInject<StubOneConstructorInjectionClass>(
            out var target, container, out _);
        Assert.That(succeeded, Is.False);
    }

    [Test]
    public void Injector_OverwriteDefaultArguments()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(0.5);
        var succeeded = ConstructorInjector.TryInject<StubWithDefaultArguments>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target?.Integer, Is.EqualTo(1));
            Assert.That(target?.Floating, Is.EqualTo(0.5));
        });
    }
    
    [DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.PublicFields | 
                                DynamicallyAccessedMemberTypes.PublicProperties)]
    private class StubWithDecimalDefaultArguments(int arg1, decimal arg2 = decimal.One)
    {
        public int Integer = arg1;

        public decimal Floating = arg2;
    }

    [Test]
    public void Injector_DefaultDecimalArgument()
    {
        var container = new InjectionContainer()
            .AddConstant(1);
        var succeeded = ConstructorInjector.TryInject<StubWithDecimalDefaultArguments>(
            out var target, container, out _);
        Assert.That(succeeded, Is.True);
        Assert.That(target?.Integer, Is.EqualTo(1));
        Assert.That(target?.Floating, Is.EqualTo(decimal.One));
    }
    
    private class StubWithKeys([Injection(Key = 1)]string key1, [Injection(Key = 2)]string key2)
    {
        public string Key1 = key1;

        public string Key2 = key2;
    }
    
    [Test]
    public void Injector_WithKeys()
    {
        var container = new InjectionContainer()
            .AddConstant("Value1", 1)
            .AddConstant("Value2", 2);
        var succeeded = ConstructorInjector.TryInject<StubWithKeys>(
            out var target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target?.Key1, Is.EqualTo("Value1"));
            Assert.That(target?.Key2, Is.EqualTo("Value2"));
        });
    }
}