﻿using System.Diagnostics.CodeAnalysis;
using Robotless.Modules.Injecting;
using Robotless.Modules.Injecting.Injectors;

namespace Robotless.Platform.Test.Modules.Injecting;

[TestFixture, TestOf(typeof(MemberInjector))]
public class TestMemberInjectorGenerator
{
    [DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.PublicFields | 
                                DynamicallyAccessedMemberTypes.PublicProperties)]
    private class StubMemberInjectionTarget
    {
        [Injection] public int NumberField = 0;

        [Injection] public int NumberProperty { get; set; } = 0;

        [Injection] public long LongField = 0;

        [Injection] public double DoubleField = 0;

        [Injection] public string StringField = "";
    }

    [Test]
    public void Injector_Create_NotNull()
    {
        var injector = MemberInjector.Get(typeof(StubMemberInjectionTarget));
        Assert.That(injector, Is.Not.Null);
    }

    [Test]
    public void Injector_InjectFields()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(2L)
            .AddConstant(1.0)
            .AddConstant("Sample");
        var sample = new StubMemberInjectionTarget();
        MemberInjector.TryInject<StubMemberInjectionTarget>(sample, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(sample.NumberField, Is.EqualTo(1));
            Assert.That(sample.DoubleField, Is.EqualTo(1.0));
            Assert.That(sample.StringField, Is.EqualTo("Sample"));
            Assert.That(sample.LongField, Is.EqualTo(2));
        });
    }
    
    private class StubMemberInjectionTargetWithRequired
    {
        [Injection] public int NumberField = 0;

        [Injection] public int NumberProperty { get; set; } = 0;

        [Injection] public long LongField = 0;

        [Injection(IsRequired = true)] public double DoubleField = 0;

        [Injection] public string StringField = "";
    }
    
    [Test]
    public void Injector_InjectFailed()
    {
        var container = new InjectionContainer()
            .AddConstant(1)
            .AddConstant(2L)
            .AddConstant("Sample");
        var sample = new StubMemberInjectionTargetWithRequired();
        var succeeded = MemberInjector.TryInject<StubMemberInjectionTargetWithRequired>(
            sample, container, out var missing);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.False);
            Assert.That(missing.Instance, Is.EqualTo(sample));
        });
    }

    [Test]
    public void Injector_InjectProperties()
    {
        var container = new InjectionContainer()
            .AddConstant(1);
        var sample = new StubMemberInjectionTarget();
        MemberInjector.TryInject<StubMemberInjectionTarget>(sample, container, out _);
        Assert.Multiple(() => { Assert.That(sample.NumberProperty, Is.EqualTo(1)); });
    }
    
    private class StubWithKeys
    {
        [Injection(Key = 1)]
        public string Key1;

        [Injection(Key = 2)]
        public string Key2;
    }
    
    [Test]
    public void Injector_WithKeys()
    {
        var container = new InjectionContainer()
            .AddConstant("Value1", 1)
            .AddConstant("Value2", 2);
        var target = new StubWithKeys();
        var succeeded = MemberInjector.TryInject<StubWithKeys>(target, container, out _);
        Assert.Multiple(() =>
        {
            Assert.That(succeeded, Is.True);
            Assert.That(target.Key1, Is.EqualTo("Value1"));
            Assert.That(target.Key2, Is.EqualTo("Value2"));
        });
    }
}