﻿using System.Collections;
using MongoDB.Bson;
using MongoDB.Bson.IO;
using Newtonsoft.Json.Bson;
using NUnit.Framework.Internal;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Serializers.Containers;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Platform.Test.Modules.Serializing.Containers;

[TestFixture]
public class TestContainerSerializer
{
    private static Randomizer Random => TestContext.CurrentContext.Random;

    [Test, TestOf(typeof(KeyValuePairSerializer<,>))]
    public void LoadAndSaveSnapshot_KeyValuePair()
    {
        var target = new KeyValuePair<int, int>(
            Random.Next(),
            Random.Next());

        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers();
        var serializer = context.GetSerializer<KeyValuePair<int, int>>();
        Assert.That(serializer, Is.Not.Null);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EqualTo(target));
    }

    private static void TestSaveAndLoadSnapshot<TContainer>(TContainer target)
        where TContainer : IEnumerable
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers();
        var serializer = context.GetSerializer<TContainer>();
        Assert.That(serializer, Is.Not.Null);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EquivalentTo(target));
    }
    
    private static void TestSaveAndLoadSnapshot<TInterface, TContainer>(TContainer target)
        where TInterface : IEnumerable
        where TContainer : TInterface
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers();
        var serializer = context.GetSerializer<TInterface>();
        Assert.That(serializer, Is.Not.Null);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EquivalentTo(target));
    }

    private static IEnumerable<int> GetRandomIntegers()
    {
        var count = Random.Next(1, 10);
#if DEBUG
        Console.WriteLine($"Sequence Length: {count}.");
#endif
        for (var index = 0; index < count; ++index)
            yield return Random.Next();
    }

    private static IEnumerable<KeyValuePair<int, int>> GetRandomKeyValuePairs()
    {
        var count = Random.Next(1, 10);
#if DEBUG
        Console.WriteLine($"Sequence Length: {count}.");
#endif
        for (var index = 0; index < count; ++index)
        {
            yield return new KeyValuePair<int, int>(Random.Next(), Random.Next());
        }
    }

    private static IEnumerable<KeyValuePair<string, int>> GetRandomStringValuePairs()
    {
        var count = Random.Next(1, 10);
#if DEBUG
        Console.WriteLine($"Sequence Length: {count}.");
#endif
        for (var index = 0; index < count; ++index)
        {
            yield return new KeyValuePair<string, int>(Random.GetString(), Random.Next());
        }
    }
    
    [Test, TestOf(typeof(EnumerableSerializer<>))]
    public void SaveAndLoadSnapshot_Value_IEnumerable()
    {
        TestSaveAndLoadSnapshot<IEnumerable<int>, StubEnumerable<int>>
            (new StubEnumerable<int>(GetRandomIntegers()));
    }

    [Test, TestOf(typeof(CollectionSerializer<>))]
    public void SaveAndLoadSnapshot_Value_ICollection()
    {
        TestSaveAndLoadSnapshot(new StubCollection<int>(GetRandomIntegers()));
    }
    
    [Test, TestOf(typeof(EnumerableSerializer<>))]
    public void SaveAndLoadSnapshot_Value_IReadOnlyCollection()
    {
        TestSaveAndLoadSnapshot<IReadOnlyCollection<int>, StubReadOnlyCollection<int>>
            (new StubReadOnlyCollection<int>(GetRandomIntegers()));
    }
    
    [Test, TestOf(typeof(ListSerializer<>))]
    public void SaveAndLoadSnapshot_Value_IList()
    {
        TestSaveAndLoadSnapshot(new List<int>(GetRandomIntegers()));
    }
    
    [Test, TestOf(typeof(ReadOnlyListSerializer<>))]
    public void SaveAndLoadSnapshot_Value_IReadOnlyList()
    {
        TestSaveAndLoadSnapshot<IReadOnlyList<int>, StubReadOnlyList<int>>
            (new StubReadOnlyList<int>(GetRandomIntegers()));
    }

    [Test, TestOf(typeof(SetSerializer<>))]
    public void SaveAndLoadSnapshot_Value_ISet()
    {
        TestSaveAndLoadSnapshot(new HashSet<int>(GetRandomIntegers()));
    }
    
    [Test, TestOf(typeof(ReadOnlySetSerializer<>))]
    public void SaveAndLoadSnapshot_Value_IReadOnlySet()
    {
        TestSaveAndLoadSnapshot<IReadOnlySet<int>, StubReadOnlySet<int>>
            (new StubReadOnlySet<int>(GetRandomIntegers()));
    }
    
    [Test, TestOf(typeof(KeyValuePairSerializer<,>))]
    public void SaveAndLoadSnapshot_Value_KeyValuePair()
    {
        var target = new KeyValuePair<int, int>(
            Random.Next(),
            Random.Next());
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers();
        var serializer = context.GetSerializer<KeyValuePair<int, int>>();
        Assert.That(serializer, Is.Not.Null);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EqualTo(target));
    }
    
    [Test, TestOf(typeof(KeyValuePairSerializer<,>))]
    public void SaveAndLoadSnapshot_Value_StringValuePair()
    {
        var target = new KeyValuePair<string, int>(
            Random.GetString(),
            Random.Next());
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers();
        var serializer = context.GetSerializer<KeyValuePair<string, int>>();
        Assert.That(serializer, Is.Not.Null);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EqualTo(target));
    }

    [Test, TestOf(typeof(DictionarySerializer<,>))]
    public void SaveAndLoadSnapshot_Value_IDictionary()
    {
        TestSaveAndLoadSnapshot(new Dictionary<int, int>(GetRandomKeyValuePairs()));
    }

    [Test, TestOf(typeof(StringDictionarySerializer<>))]
    public void SaveAndLoadSnapshot_Value_IStringDictionary()
    {
        TestSaveAndLoadSnapshot(new Dictionary<string, int>(GetRandomStringValuePairs()));
    }
    
    
}