﻿using System.Reflection;
using System.Reflection.Emit;
using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Serializers.Containers;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Platform.Test.Modules.Serializing.Generator;

[TestFixture, TestOf(typeof(SerializerGenerator))]
public class TestSerializerGenerator
{
    [Test]
    public void Create_TypeShouldNotBeNull()
    {
        var module = AssemblyBuilder
            .DefineDynamicAssembly(new AssemblyName("TestAssembly"), AssemblyBuilderAccess.Run)
            .DefineDynamicModule("TestModule");
        var generator = new SerializerGenerator();
        var type = generator.GetSerializerType(typeof(StubTargetClass));
        Assert.That(type, Is.Not.Null);
        Assert.That(type.IsAssignableTo(typeof(ISnapshotSerializer<>)
            .MakeGenericType(typeof(StubTargetClass))), Is.True);
    }

    [Test]
    public void Create_InstanceShouldNotBeNull()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithContainerSerializers()
            .WithGenerator();
        var serializer = context.GetSerializer(typeof(StubTargetClass));
        Assert.That(serializer, Is.Not.Null);
    }

    [Test]
    public void LoadSnapshot_ISnapshotSerializable_Class_ShouldInvoke()
    {
        var target = new StubCustomizedPostSerializationClass();
        var context = new SerializationContext().WithGenerator();
        var serializer = context.RequireSerializer<StubCustomizedPostSerializationClass>();
        var document = new BsonDocument();
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.LoadSnapshot(ref target, reader);
        Assert.Multiple(() =>
        {
            Assert.That(target.LoaderInvokedTimes, Is.EqualTo(1));
            Assert.That(target.SaverInvokedTimes, Is.EqualTo(0));
        });
    }

    [Test]
    public void SaveSnapshot_ISnapshotSerializable_Class_ShouldInvoke()
    {
        var target = new StubCustomizedPostSerializationClass();
        var context = new SerializationContext().WithGenerator();
        var serializer = context.RequireSerializer<StubCustomizedPostSerializationClass>();
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        serializer.SaveSnapshot(target, writer);
        Assert.Multiple(() =>
        {
            Assert.That(target.LoaderInvokedTimes, Is.EqualTo(0));
            Assert.That(target.SaverInvokedTimes, Is.EqualTo(1));
        });
    }
    
    [Test]
    public void LoadSnapshot_ISnapshotSerializable_Struct_ShouldInvoke()
    {
        var target = new StubCustomizedPostSerializationStruct();
        var context = new SerializationContext().WithGenerator();
        var serializer = context.RequireSerializer<StubCustomizedPostSerializationStruct>();
        var document = new BsonDocument();
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.LoadSnapshot(ref target, reader);
        Assert.Multiple(() =>
        {
            Assert.That(target.LoaderInvokedTimes, Is.EqualTo(1));
            Assert.That(target.SaverInvokedTimes, Is.EqualTo(0));
        });
    }

    [Test]
    public void SaveSnapshot_ISnapshotSerializable_Struct_ShouldInvoke()
    {
        var target = new StubCustomizedPostSerializationStruct();
        var context = new SerializationContext().WithGenerator();
        var serializer = context.RequireSerializer<StubCustomizedPostSerializationStruct>();
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        serializer.SaveSnapshot(target, writer);
        Assert.Multiple(() =>
        {
            Assert.That(target.LoaderInvokedTimes, Is.EqualTo(0));
            Assert.That(target.SaverInvokedTimes, Is.EqualTo(1));
        });
    }

    [Test]
    public void SelectSerializableMember_OnlyPublicReadAndWrite()
    {
        var context = new SerializationContext()
            .WithSerializer<int, Integer32Serializer>()
            .WithSerializer<double, FakeSpecificSerializer<double>>()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubMemberFilterTarget>();
        var document = new BsonDocument()
        {
            [nameof(StubMemberFilterTarget.Field)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubMemberFilterTarget.Property)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubMemberFilterTarget.GetAndInitProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubMemberFilterTarget.ConstNumberField)] = TestContext.CurrentContext.Random.NextDouble(),
            [nameof(StubMemberFilterTarget.ReadOnlyNumberField)] = TestContext.CurrentContext.Random.NextDouble(),
            [nameof(StubMemberFilterTarget.GetterOnlyProperty)] = TestContext.CurrentContext.Random.NextDouble(),
            [nameof(StubMemberFilterTarget.InitOnlyProperty)] = TestContext.CurrentContext.Random.NextDouble(),
            [nameof(StubMemberFilterTarget.PrivateSetterProperty)] = TestContext.CurrentContext.Random.NextDouble(),
            ["_privateField"] = TestContext.CurrentContext.Random.NextDouble(),
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.NewInstance(out var restoredTarget);
        Assert.DoesNotThrow(() => { serializer.LoadSnapshot(ref restoredTarget, reader); });
    }

    [Test]
    public void SaveSnapshot_Class()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();

        var serializer = context.RequireSerializer<StubTargetClass>();

        var target = new StubTargetClass()
        {
            NumberField = TestContext.CurrentContext.Random.Next(),
            NumberProperty = TestContext.CurrentContext.Random.Next(),
            GetAndInitNumberProperty = TestContext.CurrentContext.Random.Next(),
            StringField = TestContext.CurrentContext.Random.GetString(),
            StringProperty = TestContext.CurrentContext.Random.GetString()
        };

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        serializer.SaveSnapshot(target, writer);
        
        Assert.Multiple(() =>
        {
            Assert.That(document[nameof(StubTargetClass.NumberField)].AsInt32,
                Is.EqualTo(target.NumberField));
            Assert.That(document[nameof(StubTargetClass.NumberProperty)].AsInt32,
                Is.EqualTo(target.NumberProperty));
            Assert.That(document[nameof(StubTargetClass.GetAndInitNumberProperty)].AsInt32,
                Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(document[nameof(StubTargetClass.StringField)].AsString,
                Is.EqualTo(target.StringField));
            Assert.That(document[nameof(StubTargetClass.StringProperty)].AsString,
                Is.EqualTo(target.StringProperty));
        });
    }

    [Test]
    public void LoadSnapshot_Class()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetClass>();

        var document = new BsonDocument()
        {
            [nameof(StubTargetClass.NumberField)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetClass.NumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetClass.GetAndInitNumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetClass.StringField)] = TestContext.CurrentContext.Random.GetString(),
            [nameof(StubTargetClass.StringProperty)] = TestContext.CurrentContext.Random.GetString()
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.Multiple(() =>
        {
            Assert.That(restored.NumberField, 
                Is.EqualTo(document[nameof(StubTargetClass.NumberField)].AsInt32));
            Assert.That(restored.NumberProperty, 
                Is.EqualTo(document[nameof(StubTargetClass.NumberProperty)].AsInt32));
            Assert.That(restored.GetAndInitNumberProperty,
                Is.EqualTo(document[nameof(StubTargetClass.GetAndInitNumberProperty)].AsInt32));
            Assert.That(restored.StringField, 
                Is.EqualTo(document[nameof(StubTargetClass.StringField)].AsString));
            Assert.That(restored.StringProperty, 
                Is.EqualTo(document[nameof(StubTargetClass.StringProperty)].AsString));
        });
    }

    [Test]
    public void SaveSnapshot_Struct()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetStruct>();

        var target = new StubTargetStruct()
        {
            NumberField = TestContext.CurrentContext.Random.Next(),
            NumberProperty = TestContext.CurrentContext.Random.Next(),
            GetAndInitNumberProperty = TestContext.CurrentContext.Random.Next(),
            StringField = TestContext.CurrentContext.Random.GetString(),
            StringProperty = TestContext.CurrentContext.Random.GetString()
        };

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        serializer.SaveSnapshot(target, writer);
        
        Assert.Multiple(() =>
        {
            Assert.That(document[nameof(StubTargetStruct.NumberField)].AsInt32,
                Is.EqualTo(target.NumberField));
            Assert.That(document[nameof(StubTargetStruct.NumberProperty)].AsInt32,
                Is.EqualTo(target.NumberProperty));
            Assert.That(document[nameof(StubTargetStruct.GetAndInitNumberProperty)].AsInt32,
                Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(document[nameof(StubTargetStruct.StringField)].AsString,
                Is.EqualTo(target.StringField));
            Assert.That(document[nameof(StubTargetStruct.StringProperty)].AsString,
                Is.EqualTo(target.StringProperty));
        });
    }
    
    [Test]
    public void LoadSnapshot_Struct()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetStruct>();

        var document = new BsonDocument()
        {
            [nameof(StubTargetStruct.NumberField)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetStruct.NumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetStruct.GetAndInitNumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetStruct.StringField)] = TestContext.CurrentContext.Random.GetString(),
            [nameof(StubTargetStruct.StringProperty)] = TestContext.CurrentContext.Random.GetString()
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.Multiple(() =>
        {
            Assert.That(restored.NumberField, 
                Is.EqualTo(document[nameof(StubTargetStruct.NumberField)].AsInt32));
            Assert.That(restored.NumberProperty, 
                Is.EqualTo(document[nameof(StubTargetStruct.NumberProperty)].AsInt32));
            Assert.That(restored.GetAndInitNumberProperty,
                Is.EqualTo(document[nameof(StubTargetStruct.GetAndInitNumberProperty)].AsInt32));
            Assert.That(restored.StringField, 
                Is.EqualTo(document[nameof(StubTargetStruct.StringField)].AsString));
            Assert.That(restored.StringProperty, 
                Is.EqualTo(document[nameof(StubTargetStruct.StringProperty)].AsString));
        });
    }
    
    [Test]
    public void SaveAndLoadSnapshot_Class()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetClass>();

        var target = new StubTargetClass()
        {
            NumberField = TestContext.CurrentContext.Random.Next(),
            NumberProperty = TestContext.CurrentContext.Random.Next(),
            GetAndInitNumberProperty = TestContext.CurrentContext.Random.Next(),
            StringField = TestContext.CurrentContext.Random.GetString(),
            StringProperty = TestContext.CurrentContext.Random.GetString(),
        };

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.Multiple(() =>
        {
            Assert.That(restored.NumberField, Is.EqualTo(target.NumberField));
            Assert.That(restored.NumberProperty, Is.EqualTo(target.NumberProperty));
            Assert.That(restored.GetAndInitNumberProperty, Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(restored.StringField, Is.EqualTo(target.StringField));
            Assert.That(restored.StringProperty, Is.EqualTo(target.StringProperty));
        });
    }
    
    [Test]
    public void SaveAndLoadSnapshot_Struct()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetStruct>();

        var target = new StubTargetStruct()
        {
            NumberField = TestContext.CurrentContext.Random.Next(),
            NumberProperty = TestContext.CurrentContext.Random.Next(),
            GetAndInitNumberProperty = TestContext.CurrentContext.Random.Next(),
            StringField = TestContext.CurrentContext.Random.GetString(),
            StringProperty = TestContext.CurrentContext.Random.GetString(),
        };

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.Multiple(() =>
        {
            Assert.That(restored.NumberField, Is.EqualTo(target.NumberField));
            Assert.That(restored.NumberProperty, Is.EqualTo(target.NumberProperty));
            Assert.That(restored.GetAndInitNumberProperty, Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(restored.StringField, Is.EqualTo(target.StringField));
            Assert.That(restored.StringProperty, Is.EqualTo(target.StringProperty));
        });
    }

    [Test]
    public void SaveSnapshot_ChildClass()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();

        var serializer = context.RequireSerializer<StubTargetChild>();

        var target = new StubTargetChild()
        {
            NumberField = TestContext.CurrentContext.Random.Next(),
            NumberProperty = TestContext.CurrentContext.Random.Next(),
            GetAndInitNumberProperty = TestContext.CurrentContext.Random.Next(),
            StringField = TestContext.CurrentContext.Random.GetString(),
            StringProperty = TestContext.CurrentContext.Random.GetString(),
            ChildNumberField = TestContext.CurrentContext.Random.Next(),
            ChildNumberProperty = TestContext.CurrentContext.Random.Next(),
        };

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        serializer.SaveSnapshot(target, writer);
        
        Assert.Multiple(() =>
        {
            Assert.That(document[nameof(StubTargetChild.NumberField)].AsInt32,
                Is.EqualTo(target.NumberField));
            Assert.That(document[nameof(StubTargetChild.NumberProperty)].AsInt32,
                Is.EqualTo(target.NumberProperty));
            Assert.That(document[nameof(StubTargetChild.GetAndInitNumberProperty)].AsInt32,
                Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(document[nameof(StubTargetChild.StringField)].AsString,
                Is.EqualTo(target.StringField));
            Assert.That(document[nameof(StubTargetChild.StringProperty)].AsString,
                Is.EqualTo(target.StringProperty));
            Assert.That(document[nameof(StubTargetChild.ChildNumberField)].AsInt32,
                Is.EqualTo(target.ChildNumberField));
            Assert.That(document[nameof(StubTargetChild.ChildNumberProperty)].AsInt32,
                Is.EqualTo(target.ChildNumberProperty));
        });
    }
    
    [Test]
    public void LoadSnapshot_ChildClass()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetChild>();

        var document = new BsonDocument()
        {
            [nameof(StubTargetChild.NumberField)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetChild.NumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetChild.GetAndInitNumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetChild.ChildNumberField)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetChild.ChildNumberProperty)] = TestContext.CurrentContext.Random.Next(),
            [nameof(StubTargetChild.StringField)] = TestContext.CurrentContext.Random.GetString(),
            [nameof(StubTargetChild.StringProperty)] = TestContext.CurrentContext.Random.GetString()
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);

        Assert.Multiple(() =>
        {
            Assert.That(restored.NumberField, 
                Is.EqualTo(document[nameof(StubTargetChild.NumberField)].AsInt32));
            Assert.That(restored.NumberProperty, 
                Is.EqualTo(document[nameof(StubTargetChild.NumberProperty)].AsInt32));
            Assert.That(restored.GetAndInitNumberProperty,
                Is.EqualTo(document[nameof(StubTargetChild.GetAndInitNumberProperty)].AsInt32));
            Assert.That(restored.StringField, 
                Is.EqualTo(document[nameof(StubTargetChild.StringField)].AsString));
            Assert.That(restored.StringProperty, 
                Is.EqualTo(document[nameof(StubTargetChild.StringProperty)].AsString));
        });
    }

    [Test]
    public void SaveSnapshot_ValueTuple()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        var target = (TestContext.CurrentContext.Random.Next(), 
            TestContext.CurrentContext.Random.Next(),
            TestContext.CurrentContext.Random.GetString());
        context.SaveSnapshot(target, writer);
        writer.WriteEndDocument();
        Assert.Multiple(() =>
        {
            var snapshot = document["Value"]!.AsBsonDocument!;
            Assert.That(snapshot.BsonType, Is.EqualTo(BsonType.Document));
            Assert.That(snapshot["Item1"].AsInt32, Is.EqualTo(target.Item1));
            Assert.That(snapshot["Item2"].AsInt32, Is.EqualTo(target.Item2));
            Assert.That(snapshot["Item3"].AsString, Is.EqualTo(target.Item3));
        });
    }
    
    [Test]
    public void LoadSnapshot_ValueTuple()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();

        var document = new BsonDocument()
        {
            ["Value"] = new BsonDocument()
            {
                ["Item1"] = TestContext.CurrentContext.Random.Next(),
                ["Item2"] = TestContext.CurrentContext.Random.Next(),
                ["Item3"] = TestContext.CurrentContext.Random.GetString()
            }
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        var serializer = context.RequireSerializer<(int, int, string)>();
        serializer.NewInstance(out var target);
        serializer.LoadSnapshot(ref target, reader);
        Assert.Multiple(() =>
        {
            Assert.That(target.Item1, Is.EqualTo(document["Value"]["Item1"].AsInt32));
            Assert.That(target.Item2, Is.EqualTo(document["Value"]["Item2"].AsInt32));
            Assert.That(target.Item3, Is.EqualTo(document["Value"]["Item3"].AsString));
        });
    }
}