﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Serializers;
using Robotless.Modules.Serializing.Serializers.Embedded;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Platform.Test.Modules.Serializing;

[TestFixture]
public class TestPrimitiveSerializers
{
    private static (BsonDocument Document, IBsonWriter Writer) PrepareValueWriter()
    {
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        return (document, writer);
    }

    private static (BsonDocument Document, SnapshotReader Reader) PrepareValueReader<TType>(TType value)
    {
        var document = new BsonDocument
        {
            ["Value"] = BsonValue.Create(value)
        };
        var reader = new BsonDocumentReader(document);
        reader.ReadStartDocument();
        reader.ReadName("Value");
        return (document, new SnapshotReader(reader));
    }

    [Test]
    [TestCase((sbyte)42)]
    [TestCase((byte)42)]
    [TestCase(42)]
    [TestCase(42U)]
    [TestCase(42L)]
    [TestCase(42UL)]
    [TestCase(42.1F)]
    [TestCase(42.1)]
    [TestCase("Hello, World!")]
    [TestCase(true)]
    public void SavePrimitiveValue<TType>(TType value)
    {
        var (document, writer) = PrepareValueWriter();

        PrimitiveSerializerContext.Instance.SaveSnapshot(value, writer);

        writer.WriteEndDocument();

        Assert.That(document["Value"], Is.EqualTo(BsonValue.Create(value)));
    }

    [Test]
    [TestCase((sbyte)42)]
    [TestCase((byte)42)]
    [TestCase(42)]
    [TestCase(42U)]
    [TestCase(42L)]
    [TestCase(42UL)]
    [TestCase(42.1F)]
    [TestCase(42.1)]
    [TestCase("Hello, World!")]
    [TestCase(true)]
    public void LoadPrimitiveValue<TType>(TType value)
    {
        var (document, reader) = PrepareValueReader(value);

        TType restoredValue = default!;
        PrimitiveSerializerContext.Instance.LoadSnapshot(ref restoredValue, reader);

        Assert.That(restoredValue, Is.EqualTo(value));
    }

    [Test]
    [TestCase(typeof(int))]
    [TestCase(typeof(FakeSpecificSerializer<>))]
    [TestCase(typeof(PrimitiveSerializerContext))]
    public void SaveAndLoadType(Type type)
    {
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");

        PrimitiveSerializerContext.Instance.SaveSnapshot(type, writer);

        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName("Value");

        Type restoredType = default!;
        PrimitiveSerializerContext.Instance.LoadSnapshot(ref restoredType, reader);

        Assert.That(restoredType, Is.EqualTo(type));
    }

    [Test, TestOf(typeof(ArraySerializer<>))]
    public void SaveArray()
    {
        var (document, writer) = PrepareValueWriter();
        var array = new[] { 1, 2, 3, 4, 5 };

        PrimitiveSerializerContext.Instance.SaveSnapshot(array, writer);
        writer.WriteEndDocument();

        Assert.That(document["Value"], Is.EqualTo(BsonArray.Create(array)));
    }

    [Test, TestOf(typeof(ArraySerializer<>))]
    public void LoadArray()
    {
        var array = new[] { 1, 2, 3, 4, 5 };
        var (document, reader) = PrepareValueReader(array);

        int[] restoredArray = [];
        PrimitiveSerializerContext.Instance.LoadSnapshot(ref restoredArray, reader);

        Assert.That(restoredArray, Is.EqualTo(array));
    }

    [Test, TestOf(typeof(MatrixSerializer<,>))]
    public void SaveAndLoadMatrix()
    {
        var (document, writer) = PrepareValueWriter();
        var matrix = new[,] { { 1, 2, 3 }, { 4, 5, 6 } };

        PrimitiveSerializerContext.Instance.SaveSnapshot(matrix, writer);
        writer.WriteEndDocument();

        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        
        int[,] restoredMatrix = default!;
        PrimitiveSerializerContext.Instance.LoadSnapshot(ref restoredMatrix, reader);

        Assert.That(restoredMatrix, Is.EqualTo(matrix));
    }

    [Test, TestOf(typeof(ObjectSerializer))]
    public void SaveAndLoadObject()
    {
        var (document, writer) = PrepareValueWriter();
        var target = TestContext.CurrentContext.Random.Next();
        object instance = target;

        ObjectSerializer.Shared.SaveSnapshot(instance, writer);
        writer.WriteEndDocument();

        var reader = SnapshotReaderFactory.Create(document);
        reader.ReadStartDocument();
        reader.ReadName();
        
        object restored = default!;
        ObjectSerializer.Shared.LoadSnapshot(ref restored, reader);

        Assert.That(restored, Is.EqualTo(target));
    }
}