﻿using Robotless.Modules.Serializing;

namespace Robotless.Platform.Test.Modules.Serializing;

[TestFixture, TestOf(typeof(SerializationContext))]
public class TestSerializationContext
{
    [Test]
    public void SetAndGetSerializerType_SpecificSerializer()
    {
        var context = new SerializationContext()
            .WithSerializer<int, FakeSpecificSerializer<int>>()
            .WithSerializer<long, FakeSpecificSerializer<long>>();
        
        Assert.That(context.GetSerializer(typeof(int))?.GetType(), 
            Is.EqualTo(typeof(FakeSpecificSerializer<int>)));
        Assert.That(context.GetSerializer(typeof(long))?.GetType(), 
            Is.EqualTo(typeof(FakeSpecificSerializer<long>)));
    }
    
    [Test]
    public void SetAndGetSerializerType_MatchGenericTypeParameters()
    {
        var context = new SerializationContext();
        context.SetSerializer(typeof(ValueTuple<,,>), typeof(FakeGenericSerializer<,,>));
        
        Assert.That(context.GetSerializer(typeof(ValueTuple<byte, int, long>)), 
            Is.InstanceOf<FakeGenericSerializer<long, int, byte>>());
    }

    [Test]
    public void SetSerializerType_TargetTypeMismatch()
    {
        var context = new SerializationContext();
        Assert.Throws<ArgumentException>(() =>
        {
            context.SetSerializer(typeof(int), typeof(FakeSpecificSerializer<long>));
        });
    }
    
    [Test]
    public void SetSerializerType_TargetGenericDefinitionMismatch()
    {
        var context = new SerializationContext();
        Assert.Throws<ArgumentException>(() =>
        {
            context.SetSerializer(typeof(Nullable<>), typeof(FakeGenericSerializer<,,>));
        });
    }

    [Test]
    public void GetSerializer_Instantiate()
    {
        var context = new SerializationContext()
            .WithSerializer<int, FakeSpecificSerializer<int>>()
            .WithSerializer<long, FakeSpecificSerializer<long>>();
        
        Assert.That(context.GetSerializer(typeof(int)), 
            Is.TypeOf<FakeSpecificSerializer<int>>());
        Assert.That(context.GetSerializer(typeof(long)), 
            Is.TypeOf<FakeSpecificSerializer<long>>());
    }
    
    [Test]
    public void GetSerializer_ContextInjection()
    {
        var context = new SerializationContext()
            .WithSerializer<int, FakeSpecificSerializer<int>>();

        SnapshotSerializerBase<int>? serializer = null;
        
        Assert.DoesNotThrow(() =>
        {
            serializer = (SnapshotSerializerBase<int>?)context.GetSerializer<int>();
        });
        Assert.That(serializer, Is.Not.Null);
        Assert.That(serializer.Provider, Is.EqualTo(context));
    }
    
    [Test]
    public void GetSerializer_DependencyInjection()
    {
        var context = new SerializationContext()
            .WithSerializer<int, FakeSpecificSerializer<int>>()
            .WithSerializer<string, FakeSpecificSerializer<string>>(); 
        context.SetSerializer(typeof(float), typeof(FakeSpecificSerializer<float, int, string>));
        
        Assert.That(context.GetSerializer(typeof(float)), 
            Is.TypeOf<FakeSpecificSerializer<float, int, string>>());
    }
}