﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Platform.Test.Modules.Serializing;

[TestFixture, TestOf(typeof(SnapshotReader))]
public class TestSnapshotReader
{
    [Test]
    public void Locate_IterateSingleDocument()
    {
        var document = new BsonDocument()
        {
            ["A"] = 42,
            ["B"] = new BsonDocument(),
            ["C"] = "Text",
            ["D"] = new BsonArray()
            {
                1, 2, 3
            }
        };
        var reader = new BsonDocumentReader(document);
        reader.ReadStartDocument();
        
        var indexer = new SnapshotReader(reader);
        
        Assert.DoesNotThrow(() => indexer.Locate("C"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.String));
        });
        
        Assert.DoesNotThrow(() => indexer.Locate("B"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Document));
        });
        
        Assert.DoesNotThrow(() => indexer.Locate("D"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Array));
        });
        
        Assert.DoesNotThrow(() => indexer.Locate("A"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        });
    }
    
    [Test]
    public void Locate_IterateNestedDocument()
    {
        var document = new BsonDocument()
        {
            ["A"] = 42,
            ["B"] = new BsonDocument
            {
                ["B.A"] = 1,
                ["B.B"] = 2,
                ["B.C"] = 3
            },
            ["C"] = "Text",
            ["D"] = new BsonArray()
            {
                1, 2, 3
            }
        };
        var reader = new BsonDocumentReader(document);
        reader.ReadStartDocument();
        
        var indexer = new SnapshotReader(reader);
        
        Assert.DoesNotThrow(() => indexer.Locate("B"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Document));
        });
        
        reader.ReadStartDocument();
        indexer.Locate("B.A");
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        });
        indexer.SkipValue();
        indexer.Locate("B.B");
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        });
        indexer.SkipValue();
        indexer.Locate("B.C");
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        });
        indexer.SkipValue();
        reader.ReadEndDocument();
        
        Assert.DoesNotThrow(() => indexer.Locate("D"));
        Assert.Multiple(() =>
        {
            Assert.That(indexer.State, Is.EqualTo(BsonReaderState.Value));
            Assert.That(indexer.GetCurrentBsonType(), Is.EqualTo(BsonType.Array));
        });
    }

    private static SnapshotReader PrepareReader()
    {
        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("A");
        writer.WriteString("A");
        writer.WriteName("B");
        writer.WriteString("B");
        writer.WriteName("C");
        writer.WriteString("C");
        writer.WriteEndDocument();
        return new SnapshotReader(new BsonDocumentReader(document));
    }
    
    [Test]
    public void Locate_ContinuouslyInvoke()
    {
        var reader = PrepareReader();
        reader.ReadStartDocument();
        reader.Locate("A");
        reader.Locate("B");
        reader.Locate("C");
        reader.SkipValue();
        reader.ReadEndDocument();
    }
    
    [Test]
    public void Locate_DisorderInvoke()
    {
        var reader = PrepareReader();
        reader.ReadStartDocument();
        reader.Locate("B");
        reader.Locate("C");
        reader.Locate("A");
        reader.SkipValue();
        reader.ReadEndDocument();
    }

    [Test]
    public void Locate_MemoryStream()
    {
        var stream = new MemoryStream();
        var writer = new BsonBinaryWriter(stream);
        writer.WriteStartDocument();
        writer.WriteName("A");
        writer.WriteInt32(42);
        writer.WriteName("B");
        writer.WriteInt32(43);
        writer.WriteEndDocument();

        stream.Position = 0;

        var reader = new SnapshotReader(new BsonBinaryReader(stream));
        reader.ReadStartDocument();
        reader.Locate("B");
        Assert.That(reader.State, Is.EqualTo(BsonReaderState.Value));
        Assert.That(reader.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        Assert.That(reader.ReadInt32(), Is.EqualTo(43));
        reader.Locate("A");
        Assert.That(reader.State, Is.EqualTo(BsonReaderState.Value));
        Assert.That(reader.GetCurrentBsonType(), Is.EqualTo(BsonType.Int32));
        Assert.That(reader.ReadInt32(), Is.EqualTo(42));
        reader.ReadEndDocument();
    }
}