﻿using MongoDB.Bson;
using MongoDB.Bson.IO;
using Robotless.Modules.Serializing;
using Robotless.Modules.Serializing.Serializers.Primitives;
using Robotless.Modules.Serializing.Utilities;

namespace Robotless.Platform.Test.Modules.Serializing;

[TestFixture, TestOf(typeof(SnapshotSerializerBase<>))]
public class TestSnapshotSerializer
{
    [Test]
    public void LoadSnapshot_HandleNull()
    {
        var mockSerializer = new Mock<SnapshotSerializerBase<string>>(MockBehavior.Strict);

        var document = new BsonDocument
        {
            ["Value"] = BsonNull.Value
        };
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        
        // Initialized as a random value other than 0.
        var target = TestContext.CurrentContext.Random.GetString();
        
        Assert.DoesNotThrow(() =>
        {
            mockSerializer.Object.LoadSnapshot(ref target, reader);
        });
        Assert.That(target, Is.Null);
    }

    [Test]
    public void SaveSnapshot_HandleNull()
    {
        var mockSerializer = new Mock<SnapshotSerializerBase<string>>(MockBehavior.Strict);

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        
        // Initialized as a random value other than 0.
        string target = null!;
        
        Assert.DoesNotThrow(() =>
        {
            mockSerializer.Object.SaveSnapshot(target, writer);
        });
            
        Assert.That(document.Contains("Value"), Is.True);
        Assert.That(document["Value"].BsonType, Is.EqualTo(BsonType.Null));
    }

    [Test]
    public void SaveAndLoadSnapshot_ChildClass()
    {
        var context = new SerializationContext()
            .WithPrimitiveSerializers()
            .WithGenerator();
        var serializer = context.RequireSerializer<StubTargetClass>();
        StubTargetClass target = new StubTargetChild();

        var document = new BsonDocument();
        var writer = new BsonDocumentWriter(document);
        writer.WriteStartDocument();
        writer.WriteName("Value");
        serializer.SaveSnapshot(target, writer);
        writer.WriteEndDocument();
        
        Assert.Multiple(() =>
        {
            Assert.That(document["Value"].BsonType, Is.EqualTo(BsonType.Document));
            Assert.That(document["Value"]["!Type"].BsonType, Is.EqualTo(BsonType.String));
            Assert.That(document["Value"]["!Value"].BsonType, Is.EqualTo(BsonType.Document));
        });
        
        var reader = new SnapshotReader(new BsonDocumentReader(document));
        reader.ReadStartDocument();
        reader.ReadName();
        serializer.NewInstance(out var restored);
        serializer.LoadSnapshot(ref restored, reader);
        
        Assert.Multiple(() =>
        {
            Assert.That(restored, Is.InstanceOf<StubTargetChild>());
            Assert.That(restored.NumberField, Is.EqualTo(target.NumberField));
            Assert.That(restored.NumberProperty, Is.EqualTo(target.NumberProperty));
            Assert.That(restored.GetAndInitNumberProperty, Is.EqualTo(target.GetAndInitNumberProperty));
            Assert.That(((StubTargetChild)restored).ChildNumberField, 
                Is.EqualTo(((StubTargetChild)target).ChildNumberField));
            Assert.That(((StubTargetChild)restored).ChildNumberProperty, 
                Is.EqualTo(((StubTargetChild)target).ChildNumberProperty));
        });
    }
}