from datetime import datetime

from tau_trait.types import Task, Action
from tau_trait.envs.telehealth.data import load_data

def _format_currency(value: float) -> str:
    """Return a 2-decimal string for currency comparisons."""
    return f"{value:.2f}"


def _next_appointment_id(appointments: dict[str, dict]) -> str:
    """Replicate the ID generation used by schedule_appointment."""
    existing_ids = [
        int(appt_id.replace("APPT", ""))
        for appt_id in appointments
        if appt_id.startswith("APPT") and appt_id[4:].isdigit()
    ]
    new_id_num = max(existing_ids) + 1 if existing_ids else 1
    return f"APPT{new_id_num:03d}"


def _task_three_outputs() -> list[str]:
    data = load_data()
    patient = data["patients"]["maria_rodriguez_4567"]
    appointments = data["appointments"]
    existing_link = appointments["APPT007"]["meeting_link"]
    new_appointment_id = _next_appointment_id(appointments)
    insurance = patient["insurance"]["primary"]
    primary_copay = insurance["copay_primary"]
    specialist_copay = insurance["copay_specialist"]
    return [
        _format_currency(primary_copay),
        _format_currency(specialist_copay),
        existing_link,
        f"https://telehealth.healthcenter.com/room/{new_appointment_id}",
    ]





TASKS_TEST = [
    Task(
        annotator="0",
        user_id="sarah_johnson_1234",
        instruction="""
        You are Sarah Johnson, born March 15, 1985. 
        You want to schedule a follow-up appointment on Tuesday, September 23, 2025 at 10:00 AM 
        with your primary care doctor Dr. Garcia to discuss your blood pressure medication.
        """,
        actions=[
            Action(name="schedule_appointment", kwargs={"patient_id": "sarah_johnson_1234", "provider_id": "dr_garcia_primary", "date": "2025-09-23", "time": "10:00", "appointment_type": "follow_up"}),
        ],
        outputs= [],
    ),
    Task(
        annotator="1",
        user_id="david_martinez_5678",
        instruction="""
        You are David Martinez, email david.martinez@email.com. 
        You want to schedule a consultation appointment on Monday, September 22, 2025 at 2:00 PM (14:00) 
        with Dr. Smith (the cardiologist) to discuss your heart palpitations. However, if Dr. Smith is not 
        available at that exact time, you are willing to schedule with Dr. Garcia (your primary care doctor) 
        at the same time instead. You need to check both doctors' availability and schedules first.
        """,
        actions=[
            Action(name="find_patient_by_email", kwargs={"email": "david.martinez@email.com"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_smith_cardiology"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_garcia_primary"}),
            Action(name="schedule_appointment", kwargs={"patient_id": "david_martinez_5678", "provider_id": "dr_garcia_primary", "date": "2025-09-22", "time": "14:00", "appointment_type": "sick_visit"}),
        ],
        outputs=[],
    ),
    
    Task(
        annotator="2",
        user_id="daiki_sanchez_46236",
        instruction="""
        You are Daiki Sanchez (DOB 1991-05-27) checking the portal yet again. You already fired off a note earlier asking whether that cardiology telehealth consult you vaguely remember booking ever existed, so when the assistant finally answers you expect them to look you up by your portal email and sweep specifically for any cardiology visits still stuck in pending approval. If there is nothing waiting, fine—just say so and move on, but absolutely do not fabricate or auto-create a replacement without clearing it with you.

        Once that audit is done, you want to arrange a brand-new telehealth cardiology slot on Monday, April 14, 2025 at exactly 09:00, but you will only show up if the cardiologist both speaks Japanese and has at least 20 years of experience. You need those two traits together because your dad will be listening in. If even one of those requirements fails, you would rather be routed to a human scheduler and wait than settle for an English-only or junior doctor.

        You are curt, you do not let them bargain you into compromises, and if no provider satisfies both traits simultaneously you insist on being handed off to a person instead of letting them guess.
                """,
        actions=[
            Action(name="find_patient_by_email", kwargs={"email": "daikisanchez1479@example.com"}),
            Action(name="list_patient_appointments", kwargs={"patient_id": "daiki_sanchez_46236", "status_filter": "pending_approval"}),
            Action(name="list_available_providers", kwargs={"specialty": "Cardiology"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_thompson_cardiology"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_smith_cardiology"}),
            Action(name="transfer_to_human_support", kwargs={"reason": "No cardiologist meets the Japanese language plus 20-year experience requirement for the requested slot"}),
        ],
        outputs=[],
    ),
    Task(
        annotator="3",
        user_id="mira_desai_3471",
        instruction="""
        Your user id is mira_desai_3471 and your email address is mira.desai3471@gmail.com. You want to schedule an urgent telehealth consultation for May 20, 2025 ideally at 6 AM (same day appointment). You want to consult with a primary care doctor. You prefer Dr. Garcia but any primary care provider is also fine. You have 2 urgent medical concerns to discuss. You do not want insurance billing. You want to use your two telehealth credit vouchers to pay. If only one voucher can be used, you prefer using the one ending in 224. You are reactive to the agent and will not say anything that is not asked. Your medical history is in your patient profile so you do not prefer to provide it.
        """,
        actions=[
            Action(name="find_patient_by_email", kwargs={"email": "mira.desai3471@gmail.com"}),
            Action(name="get_patient_details", kwargs={"patient_id": "mira_desai_3471"}),
            Action(name="list_available_providers", kwargs={"specialty": "Primary Care"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_garcia_primary"}),
            Action(
                name="schedule_appointment",
                kwargs={
                    "patient_id": "mira_desai_3471",
                    "provider_id": "dr_garcia_primary",
                    "date": "2025-05-20",
                    "time": "06:00",
                    "appointment_type": "sick_visit",
                    "bill_insurance": False,
                    "voucher_id": "THV-224",
                },
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="4",
        user_id="carlos_martinez_5512",
        instruction="""
        Your user id is carlos_martinez_5512 and your email is carlos.martinez5512@gmail.com. All three of your kids (ages 4, 7, and 12) started vomiting six hours ago after eating the same leftovers. You need one family telehealth sick visit today May 20, 2025 at exactly 08:30 with a pediatric-certified provider who can evaluate all three children together so you only explain things once. Their medical details are already linked in the portal and you will only share extra history if the assistant insists.

        You insist on paying with the family emergency medical fund account—not insurance—and you want that noted explicitly on the booking. You are reactive: answer prompts, but do not volunteer information.
        """,
        actions=[
            Action(name="find_patient_by_email", kwargs={"email": "carlos.martinez5512@gmail.com"}),
            Action(name="get_patient_details", kwargs={"patient_id": "carlos_martinez_5512"}),
            Action(name="get_patient_details", kwargs={"patient_id": "sofia_martinez_2019"}),
            Action(name="get_patient_details", kwargs={"patient_id": "mateo_martinez_2016"}),
            Action(name="get_patient_details", kwargs={"patient_id": "isabella_martinez_2011"}),
            Action(name="list_available_providers", kwargs={"specialty": "Pediatrics"}),
            Action(name="get_provider_details", kwargs={"provider_id": "dr_nguyen_pediatrics"}),
            Action(
                name="schedule_appointment",
                kwargs={
                    "patient_id": "carlos_martinez_5512",
                    "provider_id": "dr_nguyen_pediatrics",
                    "date": "2025-05-20",
                    "time": "08:30",
                    "appointment_type": "sick_visit",
                    "bill_insurance": False,
                },
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="5",
        user_id="maria_rodriguez_4567",
        instruction="""
        You are Maria Rodriguez (DOB 1990-12-03) catching up on that telehealth to-do list. The mid-December cardiology video visit with Dr. Smith (the one stamped APPT009 at 4 PM) never cleared insurance, so check with the assistant who you are, see whether that slot is still stuck in pending land, and only if it hasn’t been approved yet do you have them remove it so nothing lingers. Once that old booking is out of the way, you want a fresh specialist telehealth consult with whichever cardiologist has the deepest experience—start with Dr. Margaret Thompson—and you only want it if she can take you exactly at 09:00 on Tuesday, January 23, 2025. If she can’t or that time is gone, fall back to Dr. Robert Smith at that same moment; if neither doctor can commit to 09:00, you leave everything canceled and just report back. No matter how it ends, you expect to hear the insurance copay and see the telehealth link for whatever appointment winds up surviving. Keep the tone polite but brisk and expect the assistant to double-check availability instead of guessing.
        """,
        actions=[
            Action(
                name="find_patient_by_name_dob",
                kwargs={
                    "first_name": "Maria",
                    "last_name": "Rodriguez",
                    "date_of_birth": "1990-12-03",
                },
            ),
            Action(
                name="list_patient_appointments",
                kwargs={
                    "patient_id": "maria_rodriguez_4567",
                    "status_filter": "pending_approval",
                },
            ),
            Action(
                name="cancel_appointment",
                kwargs={"appointment_id": "APPT009"},
            ),
            Action(
                name="list_available_providers",
                kwargs={"specialty": "Cardiology"},
            ),
            Action(
                name="get_provider_details",
                kwargs={"provider_id": "dr_thompson_cardiology"},
            ),
            Action(
                name="schedule_appointment",
                kwargs={
                    "patient_id": "maria_rodriguez_4567",
                    "provider_id": "dr_thompson_cardiology",
                    "date": "2025-01-23",
                    "time": "09:00",
                    "appointment_type": "specialist_consultation",
                },
            ),
            Action(
                name="get_patient_details",
                kwargs={"patient_id": "maria_rodriguez_4567"},
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="6",
        user_id="heather_collins_48201",
        instruction="""
        You are Heather Collins, born April 19, 1982, dialing in from Grand Rapids. Your portal email is heather.collins82@gmail.com. You want the prescriptions from your last four telehealth encounters and, for each medication, the cheapest Indian supplier the hospital catalog lists (company, brand, USD price). If a medication lacks an Indian supplier, you expect a blunt "no Indian supplier" answer. Once you have the information, you want each prescription record updated to reflect the selected supplier details so the pharmacy team can start sourcing immediately. You are terse, impatient, and do not want small talk.
        """,
        actions=[
            Action(
                name="find_patient_by_email",
                kwargs={"email": "heather.collins82@gmail.com"},
            ),
            Action(
                name="list_patient_medical_records",
                kwargs={"patient_id": "heather_collins_48201", "limit": 4},
            ),
            Action(name="get_medical_record", kwargs={"record_id": "REC008"}),
            Action(name="get_medical_record", kwargs={"record_id": "REC007"}),
            Action(name="get_medical_record", kwargs={"record_id": "REC006"}),
            Action(name="get_medical_record", kwargs={"record_id": "REC005"}),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Fluticasone Inhaler", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC008",
                    "medication": "Fluticasone Inhaler",
                    "supplier_company": "Lotus Respiratory (India)",
                    "brand_name": "Flohale",
                    "price_usd": 7.1,
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Montelukast", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC007",
                    "medication": "Montelukast",
                    "supplier_company": "Lotus Breath (India)",
                    "brand_name": "BreathFree",
                    "price_usd": 3.05,
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Ezetimibe", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC006",
                    "medication": "Ezetimibe",
                    "supplier_company": "Lucknow Lipids (India)",
                    "brand_name": "Ezeswift",
                    "price_usd": 3.25,
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Atorvastatin", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC005",
                    "medication": "Atorvastatin",
                    "supplier_company": "VedaRx Labs (India)",
                    "brand_name": "Atorveeda",
                    "price_usd": 4.05,
                },
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="7",
        user_id="amara_osei_8834",
        instruction="""
        Your user id is amara_osei_8834 and your email is amara.osei8834@yahoo.com. Your usual insulin pens are on nationwide backorder and you have three days of supply left. You need an emergency same-day telehealth consult on Tuesday, May 20, 2025 with an endocrinologist who can source alternative insulin formulations. You insist the visit be paid from your diabetes emergency account (note it explicitly) and you remain terse unless asked direct questions. Your diabetes management plan is already in your records and you do not want to repeat it. After confirming the endocrinologist, you expect your current prescriptions to be updated with the cheapest Indian suppliers the hospital catalog lists so the pharmacy can act immediately.
        """,
        actions=[
            Action(
                name="find_patient_by_email",
                kwargs={"email": "amara.osei8834@yahoo.com"},
            ),
            Action(
                name="list_patient_medical_records",
                kwargs={"patient_id": "amara_osei_8834", "limit": 4},
            ),
            Action(name="get_medical_record", kwargs={"record_id": "REC010"}),
            Action(name="get_medical_record", kwargs={"record_id": "REC009"}),
            Action(
                name="list_available_providers",
                kwargs={"specialty": "Endocrinology"},
            ),
            Action(
                name="get_provider_details",
                kwargs={"provider_id": "dr_singh_endocrinology"},
            ),
            Action(
                name="schedule_appointment",
                kwargs={
                    "patient_id": "amara_osei_8834",
                    "provider_id": "dr_singh_endocrinology",
                    "date": "2025-05-20",
                    "time": "13:30",
                    "appointment_type": "specialist_consultation",
                    "bill_insurance": False,
                    "payment_notes": "Charge diabetes emergency account",
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Insulin Glargine", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC010",
                    "medication": "Insulin Glargine",
                    "supplier_company": "SunPharm Endocrine",
                    "brand_name": "Basargine",
                    "price_usd": 21.6,
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Insulin Lispro", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC010",
                    "medication": "Insulin Lispro",
                    "supplier_company": "Bengal EndoCare",
                    "brand_name": "SwiftLis",
                    "price_usd": 19.2,
                },
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="8",
        user_id="mira_desai_3471",
        instruction="""
        Your user id is mira_desai_3471 and your email address is mira.desai3471@gmail.com. Overnight triage auto-booked you for an 11:00 AM pending virtual slot on May 20, 2025, but you need a same-day primary care consult much earlier—ideally 06:00—with whoever can review two urgent concerns before work. You prefer Dr. Garcia if the slot is open, otherwise any primary-care doctor will do. You refuse insurance billing; charge the visit to your telehealth credit vouchers and, if only one applies, use the one ending in 224 and note it. You only answer direct questions and do not want to rehash your medical history since it is already on file.
        """,
        actions=[
            Action(
                name="find_patient_by_email",
                kwargs={"email": "mira.desai3471@gmail.com"},
            ),
            Action(
                name="list_patient_appointments",
                kwargs={"patient_id": "mira_desai_3471", "status_filter": "pending_approval"},
            ),
            Action(
                name="cancel_appointment",
                kwargs={"appointment_id": "APPT027"},
            ),
            Action(
                name="list_available_providers",
                kwargs={"specialty": "Primary Care"},
            ),
            Action(
                name="get_provider_details",
                kwargs={"provider_id": "dr_garcia_primary"},
            ),
            Action(
                name="schedule_appointment",
                kwargs={
                    "patient_id": "mira_desai_3471",
                    "provider_id": "dr_garcia_primary",
                    "date": "2025-05-20",
                    "time": "06:00",
                    "appointment_type": "sick_visit",
                    "bill_insurance": False,
                    "voucher_id": "THV-224",
                    "payment_notes": "Apply telehealth voucher ending 224 for urgent dawn consult",
                },
            ),
        ],
        outputs=[],
    ),
    Task(
        annotator="9",
        user_id="nina_park_8020",
        instruction="""
        You are nina_park_8020 (zip 28236), email nina.park8020@gmail.com. During your remote physical-therapy session you were issued a medium resistance band set and a firm foam massage roller. Your therapist now wants you using heavier resistance and a cushioned roller. Retrieve the latest telehealth kit record to confirm the current items, then update both prescriptions with the lowest-cost Indian catalog options that meet those preferences.
        """,
        actions=[
            Action(
                name="find_patient_by_email",
                kwargs={"email": "nina.park8020@gmail.com"},
            ),
            Action(
                name="list_patient_medical_records",
                kwargs={"patient_id": "nina_park_8020", "limit": 1},
            ),
            Action(name="get_medical_record", kwargs={"record_id": "REC011"}),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Resistance Band Set", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC011",
                    "medication": "Resistance Band Set",
                    "supplier_company": "Mumbai Active Care",
                    "brand_name": "Active Care Heavy",
                    "price_usd": 17.6,
                },
            ),
            Action(
                name="list_medication_suppliers",
                kwargs={"medication": "Foam Massage Roller", "country_filter": "India", "limit": 1},
            ),
            Action(
                name="update_prescription_supplier",
                kwargs={
                    "record_id": "REC011",
                    "medication": "Foam Massage Roller",
                    "supplier_company": "Kerala Wellness Supplies",
                    "brand_name": "Kerala Soft Relief",
                    "price_usd": 14.9,
                },
            ),
        ],
        outputs=[],
    ),
    # Task(
    #     annotator="2",
    #     user_id="maria_rodriguez_4567",
    #     instruction="""
    #     You are Maria Rodriguez, born December 3, 1990. You have a complex scheduling situation:
        
    #     1. You currently have appointment APPT007 with Dr. Garcia on Monday, January 22nd 2025 at 9:00 AM
    #     2. You need to reschedule it because you have a work conflict, but you can only meet on the same day (Monday) at 3:00 PM or later.
    #     3. You also need to schedule a separate specialist cardiology consultation for your new heart palpitations, preferably with the most experienced cardiologist available
    #     4. The cardiology appointment must be on Tuesday, January 23rd 2025 at exactly 09:00 (9:00 AM) so it is finished before work
    #     5. You want to verify your insurance copays for both appointments before confirming
    #     7. You need to get the meeting links for both final appointments
        
    #     Complete ALL requirements in the correct order, making decisions based on provider availability and experience.
    #     """,
    #     actions=[
    #         Action(name="reschedule_appointment", kwargs={"appointment_id": "APPT007", "new_date": "2025-01-22", "new_time": "15:00"}),
    #         Action(name="schedule_appointment", kwargs={"patient_id": "maria_rodriguez_4567", "provider_id": "dr_thompson_cardiology", "date": "2025-01-23", "time": "09:00", "appointment_type": "specialist_consultation"}),
    #     ],
    #     outputs=_task_three_outputs(),
    # ),
    # Task(
    #     annotator="3",
    #     user_id="maria_rodriguez_4567",
    #     instruction="""
    #     You are Maria Rodriguez, born December 3, 1990. Start by confirming your identity with your
    #     full name and date of birth. Retrieve your telehealth appointment history and locate the
    #     cardiology telehealth visit from Sunday, December 15, 2024 at 4:00 PM that still shows as
    #     pending approval and is now past due. Cancel that appointment before scheduling anything new.

    #     After the pending visit is cancelled, schedule a dermatologist follow-up with Dr. Lee on
    #     Tuesday, February 4, 2025 at 10:00 (10 AM) and ensure there are no conflicts. Once it is
    #     booked, double-check the appointment details and verify that the copay shown matches the
    #     specialist copay in your patient profile.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_name_dob",
    #             kwargs={"first_name": "Maria", "last_name": "Rodriguez", "date_of_birth": "1990-12-03"},
    #         ),
    #         Action(
    #             name="list_patient_appointments",
    #             kwargs={"patient_id": "maria_rodriguez_4567"},
    #         ),
    #         Action(
    #             name="cancel_appointment",
    #             kwargs={"appointment_id": "APPT009"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "maria_rodriguez_4567",
    #                 "provider_id": "dr_lee_dermatology",
    #                 "date": "2025-02-04",
    #                 "time": "10:00",
    #                 "appointment_type": "follow_up",
    #             },
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT010"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "maria_rodriguez_4567"},
    #         ),
    #     ],
    #     outputs=["45.00", "https://telehealth.healthcenter.com/room/APPT010"],
    # ),
    # Task(
    #     annotator="4",
    #     user_id="sarah_johnson_1234",
    #     instruction="""
    #     You are Sarah Johnson, email sarah.johnson@email.com. Authenticate yourself using your email.
    #     Review only your pending telehealth appointments to locate the cardiology consult that was set
    #     for Thursday, January 18, 2024 at 3:00 PM and is still waiting on approval. Cancel that pending
    #     appointment before making any schedule changes.

    #     Once the cardiology visit is cancelled, double-check Dr. Garcia's availability to make sure he
    #     has an 11:00 AM slot open on Monday, January 15, 2024. Move your routine primary-care visit
    #     (APPT001) on that day from 9:00 AM to 11:00 AM. After rescheduling, pull the updated appointment
    #     details so you can relay the meeting link, and confirm that the listed copay for the visit still
    #     matches the primary-care copay in your patient profile.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_email",
    #             kwargs={"email": "sarah.johnson@email.com"},
    #         ),
    #         Action(
    #             name="list_patient_appointments",
    #             kwargs={"patient_id": "sarah_johnson_1234", "status_filter": "pending_approval"},
    #         ),
    #         Action(
    #             name="cancel_appointment",
    #             kwargs={"appointment_id": "APPT004"},
    #         ),
    #         Action(
    #             name="get_provider_details",
    #             kwargs={"provider_id": "dr_garcia_primary"},
    #         ),
    #         Action(
    #             name="reschedule_appointment",
    #             kwargs={"appointment_id": "APPT001", "new_date": "2024-01-15", "new_time": "11:00"},
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT001"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "sarah_johnson_1234"},
    #         ),
    #     ],
    #     outputs=["25.00", "https://telehealth.healthcenter.com/room/APPT001"],
    # ),
    # Task(
    #     annotator="5",
    #     user_id="emily_chen_9012",
    #     instruction="""
    #     You are Emily Chen, born July 8, 1992. Authenticate with your full name and date of birth. Review
    #     the medical record from your January 17, 2024 dermatology visit to confirm the treatment plan that
    #     was recommended. After verifying the plan, check Dr. Patel's availability for that same day and
    #     move your appointment (APPT003) from 11:00 AM to the earliest available afternoon slot.

    #     Once the appointment is rescheduled, retrieve the updated appointment details and confirm that the
    #     meeting link is still correct. Finally, reference your patient profile to make sure the copay listed
    #     for the visit aligns with your specialist copay.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_name_dob",
    #             kwargs={"first_name": "Emily", "last_name": "Chen", "date_of_birth": "1992-07-08"},
    #         ),
    #         Action(
    #             name="get_medical_record",
    #             kwargs={"record_id": "REC003"},
    #         ),
    #         Action(
    #             name="get_provider_details",
    #             kwargs={"provider_id": "dr_patel_dermatology"},
    #         ),
    #         Action(
    #             name="reschedule_appointment",
    #             kwargs={"appointment_id": "APPT003", "new_date": "2024-01-17", "new_time": "14:00"},
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT003"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "emily_chen_9012"},
    #         ),
    #     ],
    #     outputs=["Hydrocortisone 1% cream", "40.00", "https://telehealth.healthcenter.com/room/APPT003"],
    # ),
    # Task(
    #     annotator="6",
    #     user_id="linda_parker_8899",
    #     instruction="""
    #     You are Linda Parker, email linda.parker@email.com. You need a cardiology telehealth slot the
    #     week of March 17, 2025, but your manager only releases you in the mornings unless nothing is
    #     available. Follow your tiered plan exactly and attempt each option in order, even if you think it
    #     will be unavailable. Stay focused on cardiology providers (Dr. Thompson and Dr. Smith):

    #     1. Try Dr. Thompson (the senior cardiologist) at 9:00 AM on Tuesday, March 18, 2025.
    #     2. If that slot is already taken, immediately check Dr. Smith for a 10:00 AM appointment the
    #        same morning.
    #     3. If neither morning works, abandon Tuesday and book a Thursday afternoon visit (after 1 PM)
    #        with whichever cardiologist has at least 20 years of experience.

    #     As you work through those options, sanity-check the providers' schedules and experience levels,
    #     and keep me updated on which branch you end up using. Once you lock something in, pull the
    #     appointment details and confirm that the meeting link (include at least the base URL) and
    #     specialist copay align with my
    #     profile. I also want to hear which cardiologist ultimately satisfied the 20+ year requirement.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_email",
    #             kwargs={"email": "linda.parker@email.com"},
    #         ),
    #         Action(
    #             name="get_provider_details",
    #             kwargs={"provider_id": "dr_thompson_cardiology"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "linda_parker_8899",
    #                 "provider_id": "dr_thompson_cardiology",
    #                 "date": "2025-03-18",
    #                 "time": "09:00",
    #                 "appointment_type": "specialist_consultation",
    #             },
    #         ),
    #         Action(
    #             name="get_provider_details",
    #             kwargs={"provider_id": "dr_smith_cardiology"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "linda_parker_8899",
    #                 "provider_id": "dr_smith_cardiology",
    #                 "date": "2025-03-18",
    #                 "time": "10:00",
    #                 "appointment_type": "specialist_consultation",
    #             },
    #         ),
    #         Action(
    #             name="list_available_providers",
    #             kwargs={"specialty": "Cardiology"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "linda_parker_8899",
    #                 "provider_id": "dr_thompson_cardiology",
    #                 "date": "2025-03-20",
    #                 "time": "14:00",
    #                 "appointment_type": "specialist_consultation",
    #             },
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT013"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "linda_parker_8899"},
    #         ),
    #     ],
    #     outputs=["Dr. Thompson", "55.00", "https://telehealth.healthcenter.com/room/"],
    # ),
    # Task(
    #     annotator="7",
    #     user_id="noah_ito_98187",
    #     instruction="""
    #     You are Noah Ito, born August 19, 1989, ZIP code 98187. You're impatient and give requests in
    #     pieces. Handle each step in order:

    #     1. Authenticate with your full name and date of birth.
    #     2. You just booked APPT014 (Dr. Garcia) and APPT015 (Dr. Smith). Without me repeating the address,
    #        peek at my patient profile so you remember the New York contact details before adjusting anything.
    #     3. Move APPT014 to the latest morning slot Dr. Garcia has on Tuesday, April 8, 2025 (no afternoons).
    #     4. Clean up APPT015 by canceling it, then try to grab Wednesday, April 9 at 10:00 AM with
    #        Dr. Thompson. If she is already taken, fall back to the first Thursday slot after 1:00 PM with a
    #        cardiologist who has at least 20 years of experience.
    #     5. Once the new cardiology appointment is confirmed, pull the appointment details and make sure the
    #        meeting link and specialist copay match what my insurance profile says. Also tell me explicitly
    #        which cardiologist satisfied the 20+ year requirement.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_name_dob",
    #             kwargs={"first_name": "Noah", "last_name": "Ito", "date_of_birth": "1989-08-19"},
    #         ),
    #         Action(
    #             name="list_patient_appointments",
    #             kwargs={"patient_id": "noah_ito_98187"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "noah_ito_98187"},
    #         ),
    #         Action(
    #             name="reschedule_appointment",
    #             kwargs={"appointment_id": "APPT014", "new_date": "2025-04-08", "new_time": "11:00"},
    #         ),
    #         Action(
    #             name="cancel_appointment",
    #             kwargs={"appointment_id": "APPT015"},
    #         ),
    #         Action(
    #             name="get_provider_details",
    #             kwargs={"provider_id": "dr_thompson_cardiology"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "noah_ito_98187",
    #                 "provider_id": "dr_thompson_cardiology",
    #                 "date": "2025-04-09",
    #                 "time": "10:00",
    #                 "appointment_type": "specialist_consultation",
    #             },
    #         ),
    #         Action(
    #             name="list_available_providers",
    #             kwargs={"specialty": "Cardiology"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "noah_ito_98187",
    #                 "provider_id": "dr_thompson_cardiology",
    #                 "date": "2025-04-10",
    #                 "time": "14:00",
    #                 "appointment_type": "specialist_consultation",
    #             },
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT017"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "noah_ito_98187"},
    #         ),
    #     ],
    #     outputs=["Dr. Thompson", "60.00", "https://telehealth.healthcenter.com/room/APPT017"],
    # ),
    # Task(
    #     annotator="9",
    #     user_id="olivia_martin_4433",
    #     instruction="""
    #     You are Olivia Martin, born February 11, 1994. You're juggling spring allergies, asthma tune-ups,
    #     and a pharmacist consult. Piece this together without missing the allergy warning:

    #     - Authenticate me via full name and DOB first.
    #     - I just pinged the portal asking for two refills: the rescue albuterol inhaler and that
    #       sulfamethoxazole course my coworker swears by. Before you promise anything, dig up my most
    #       recent virtual visit notes from late March (the ones Dr. Garcia wrote) and double-check the
    #       allergy section in my profile.
    #     - If the notes hint at any conflict, refuse the bad refill explicitly, but greenlight the safe one.
    #       I still want a concrete path forward, not just "no".
    #     - Then line up a telehealth pharmacist check-in with PharmD Jones on Thursday, April 3, 2025 at
    #       14:30 Pacific (she handles med coaching for our clinic). Make sure nothing else I have overlaps.
    #     - Once that's booked, send me the appointment specifics and quote the copay pulled straight from my
    #       insurance info. Remember: I only trust answers that connect the dots between the allergy warning,
    #       the approved refill, and the pharmacist session link.
    #     """,
    #     actions=[
    #         Action(
    #             name="find_patient_by_name_dob",
    #             kwargs={"first_name": "Olivia", "last_name": "Martin", "date_of_birth": "1994-02-11"},
    #         ),
    #         Action(
    #             name="get_medical_record",
    #             kwargs={"record_id": "REC004"},
    #         ),
    #         Action(
    #             name="get_patient_details",
    #             kwargs={"patient_id": "olivia_martin_4433"},
    #         ),
    #         Action(
    #             name="list_patient_appointments",
    #             kwargs={"patient_id": "olivia_martin_4433"},
    #         ),
    #         Action(
    #             name="schedule_appointment",
    #             kwargs={
    #                 "patient_id": "olivia_martin_4433",
    #                 "provider_id": "pharmacist_jones",
    #                 "date": "2025-04-03",
    #                 "time": "14:30",
    #                 "appointment_type": "sick_visit",
    #             },
    #         ),
    #         Action(
    #             name="get_appointment_details",
    #             kwargs={"appointment_id": "APPT019"},
    #         ),
    #     ],
    #     outputs=["sulfa allergy", "Albuterol", "https://telehealth.healthcenter.com/room/"],
    # ),
]
