#!/bin/bash

#SBATCH --job-name=ratio
#SBATCH --output=ratio_e_%A_%a.out
#SBATCH --error=ratio_e_%A_%a.err
#SBATCH --array=1-55 # Adjust the range based on the number of tasks

#SBATCH --partition=gpu
#SBATCH --qos=gpu_normal

#SBATCH --nodes=1
#SBATCH --cpus-per-task=12
#SBATCH --mem=32G
#SBATCH --ntasks=1
#SBATCH --time=6:00:00
#SBATCH --nice=10000

datasets=("ENZYMES") 
# Calculate the index of the current task in the array
task_index=$((SLURM_ARRAY_TASK_ID - 1))

# List of Python scripts to run
python_scripts=(
    "run_mag_pool_edge.py --metric diffusion_distance --mag_method spread"
    "run_mag_pool_edge.py --metric diffusion_distance --mag_method cholesky"   
    "run_mincut.py"
    "run_diffpool.py"
    "run_graclus.py"
    "run_ndp.py"
    "run_nmf.py"
    "run_topk.py"
    "run_sagpool.py"
)

ratios=(
    "0.125"
    "0.25"
    "0.5"
)

models=(
    "GIN"
    "GNN"
)

# Calculate the total number of tasks
total_tasks=$(( ${#datasets[@]} * ${#python_scripts[@]} * ${#ratios[@]} * ${#models[@]} ))

# Ensure the task index is within bounds
if [ "$task_index" -ge "$total_tasks" ]; then
    echo "Task index $task_index is out of range. No task to run."
    exit 1
fi

echo "Task index $task_index "

# Determine the dataset and script for the current task
dataset_index=$(( task_index / (${#python_scripts[@]} * ${#ratios[@]} * ${#models[@]} ) ))
script_index=$(( (task_index / (${#ratios[@]} * ${#models[@]} )) % ${#python_scripts[@]} ))
ratio_index=$(( (task_index / ${#models[@]} ) % ${#ratios[@]} ))
model_index=$(( task_index % ${#models[@]} ))

echo "$dataset_index"

echo "$script_index"

current_dataset=${datasets[$dataset_index]}
current_script=${python_scripts[$script_index]}

# Print task information
echo "Running task $SLURM_ARRAY_TASK_ID with dataset: $current_dataset and script: $current_script and ratio: ${ratios[$ratio_index]} and model: ${models[$model_index]}"

# Run the Python script with the dataset
source $(poetry env info --path)/bin/activate
python $current_script --dataset $current_dataset --runs 10 --batch-size 32 --model ${models[$model_index]} --ratio ${ratios[$ratio_index]}
