import os
import numpy as np
import torch
import torch.utils.data as data
import torchvision.transforms as transforms
from PIL import Image
# from utils import text_helper
import utils
from utils import VocabDict
import random

class QuestionDataset(data.Dataset):

    def __init__(self, input_dir, input_vqa, max_qst_length=30):
        self.input_dir = input_dir
        self.vqa = np.load(input_dir+'/'+input_vqa, allow_pickle=True)
        self.qst_vocab = VocabDict(input_dir+'/vocab_questions.txt')
        self.max_qst_length = max_qst_length

    def __getitem__(self, idx):

        vqa = self.vqa
        qst_vocab = self.qst_vocab
        max_qst_length = self.max_qst_length


        # qst2idc = np.array(len(vqa[idx]['question_tokens'])+2) #np.array([qst_vocab.word2idx('<pad>')]* max_qst_length)  # padded with '<pad>' in 'ans_vocab'
        qst2idc = np.array([qst_vocab.word2idx('<start>')] + [qst_vocab.word2idx(w) for w in vqa[idx]['question_tokens']] + [qst_vocab.word2idx('<end>')])
        qst_len = len(qst2idc)
        label = 0
        qst2idc = torch.from_numpy(qst2idc).clone()
        if random.randint(0,1):
            # print()
            # gen = [qst_vocab.idx2word(w) for w in qst2idc if w not in {qst_vocab.word2idx('<end>'), qst_vocab.word2idx('<start>'), qst_vocab.word2idx('<pad>')}] 
            # print(" ".join(gen), label)
            qst2idc = self.shuffle(qst2idc, qst_len)
            label = 1
            # gen = [qst_vocab.idx2word(w) for w in qst2idc if w not in {qst_vocab.word2idx('<end>'), qst_vocab.word2idx('<start>'), qst_vocab.word2idx('<pad>')}] 
            # print(" ".join(gen), label)

        return qst2idc, label

    def shuffle(self, qst2idc, len_):
        target1 = random.randint(1,len_-2)
        while(1):
            target2 = random.randint(1,len_-2)
            if target2 != target1:
                break
        tmp = qst2idc[target1].clone()
        qst2idc[target1] = qst2idc[target2].clone()
        qst2idc[target2] = tmp
        return qst2idc        

    def __len__(self):

        return len(self.vqa)

def collate_fn_VQA(data):
    """Create mini-batches of (image, caption)

    Custom collate_fn for torch.utils.data.DataLoader is necessary for patting captions

    :param data: list; (image, caption) tuples
            - image: tensor;    3 x 256 x 256
            - caption: tensor;  1 x length_caption

    Return: mini-batch
    :return images: tensor;     batch_size x 3 x 256 x 256
    :return padded_captions: tensor;    batch_size x length
    :return caption_lengths: list;      lenghths of actual captions (without padding)
    """

    # sort data by caption length
    data.sort(key=lambda x: len(x[0]), reverse=True)
    captions, labels = zip(*data)


    # Merge captions
    caption_lengths = [len(caption) for caption in captions]

    # zero-matrix num_captions x caption_max_length
    padded_captions = torch.zeros(len(captions), max(caption_lengths)).long()

    # fill the zero-matrix with captions. the remaining zeros are padding
    for ix, caption in enumerate(captions):
        end = caption_lengths[ix]
        padded_captions[ix, :end] = caption[:end]

    return padded_captions, torch.Tensor(list(labels)).to(torch.long)

def get_data_loader(input_dir, input_vqa_train, input_vqa_valid, max_qst_length, batch_size, num_workers):


    qst_dataset = {
        'train': QuestionDataset(
            input_dir=input_dir,
            input_vqa=input_vqa_train,
            max_qst_length=max_qst_length),
        'valid': QuestionDataset(
            input_dir=input_dir,
            input_vqa=input_vqa_valid,
            max_qst_length=max_qst_length)}

    data_loader = {
        phase: torch.utils.data.DataLoader(
            dataset=qst_dataset[phase],
            batch_size=batch_size,
            shuffle=True,
            num_workers=num_workers,collate_fn=collate_fn_VQA,drop_last=True)
        for phase in ['train', 'valid']}

    return data_loader