#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "linalg.h"

// Static functions
static double house(double *x, double *v, int m);
static void house_qr(double **A, double **QR, int m, int n);
static int house_qr_pivot(double **A, double **QR, int *piv, int m, int n);
static int COD(double **A, double **U, double **T, double **V, int m, int n);

/* ------------------------------------------------------------
Vector operations
-------------------------------------------------------------*/
//  Create a new vector of size n
double *new_vec(int n)
{
	double *u;
	int i = 0;

	u = (double *)malloc(n * sizeof(double));
	for (i = 0; i < n; i++)
	{
		u[i] = 0;
	}

	return u;
}

// Free the vector u
void free_vec(double *u)
{
	free(u);
	return;
}

// Make a vector u[n] zero
void vec_zero(double *u, int n)
{
	int i = 0;

	for (i = 0; i < n; i++)
	{
		u[i] = 0;
	}
	return;
}

// Copy a vector u[n] to v[n]
void vec_cpy(double *u, double *v, int n)
{
	int i = 0;

	for (i = 0; i < n; i++)
	{
		v[i] = u[i];
	}
	return;
}

// Linear combination w = au + bv of vectors u and v
// a, b: scalar, u[n], v[n]: vectors
void vec_lincom(double *u, double *v, double a, double b, double *w, int n)
{
	int i = 0;

	for (i = 0; i < n; i++)
	{
		w[i] = a * u[i] + b * v[i];
	}

	return;
}

// Return the inner product of vectors u[n] and v[n]
double vec_innerprod(double *u, double *v, int n)
{
	int i = 0;
	double ret = 0;

	for (i = 0; i < n; i++)
	{
		ret += u[i] * v[i];
	}
	return ret;
}

// Read a vector of size n from a file
double *vec_fread(FILE *fp, int n)
{
	int i = 0;
	double *ret;

	ret = new_vec(n);

	for (i = 0; i < n; i++)
	{
		fscanf(fp, "%lf", &(ret[i]));
	}
	rewind(fp);

	return ret;
}

// Write a vector of size n to a file
void vec_fwrite(FILE *fp, double *u, int n){
	int i = 0;

	for(i=0; i<n; i++){
		fprintf(fp, "%.16lf\n", u[i]);
	}

	return;
}

/* ------------------------------------------------------------
Matrix operations
-------------------------------------------------------------*/
// Create a new matrix of size mXn
double **new_mat(int m, int n)
{
	double **A;
	int i = 0;

	A = (double **)malloc(m * sizeof(double *));
	for (i = 0; i < m; i++)
	{
		A[i] = new_vec(n);
	}

	return A;
}

// Free the mXn matrix A
void free_mat(double **A, int m)
{
	int i = 0;
	for (i = 0; i < m; i++)
	{
		free_vec(A[i]);
	}
	free(A);
	return;
}

// Make a matrix A[m][n] zero
void mat_zero(double **A, int m, int n)
{
	int i = 0;

	for (i = 0; i < m; i++)
	{
		vec_zero(A[i], n);
	}
	return;
}

// Copy a matrix A[m][n] to B[m][n]
void mat_cpy(double **A, double **B, int m, int n)
{
	int i = 0;

	for (i = 0; i < m; i++)
	{
		vec_cpy(A[i], B[i], n);
	}
	return;
}

// Compute the matrix-vector product Ax and store the result into y.
void mat_vec_mult(double **A, double *x, double *y, int m, int n){
	int i = 0, j = 0;
	double *temp = (double *) malloc(m * sizeof(double));

	for(i=0; i<m; i++){
		temp[i] = 0;
		for(j=0; j<n; j++){
			temp[i] += A[i][j]*x[j];
		}
	}

	for(i=0; i<m; i++){
		y[i] = temp[i];
	}

	free(temp);
	return;
}

// Solve the linear system Ax = b directly. (Gaussian elimination with partial pivoting)
void mat_solve(double **A, double *b, double *x, int n){
	int  i = 0, j = 0, k = 0, pivot = 0;
	double temp = 0;
	double **A_temp, *b_temp;

	A_temp = new_mat(n, n);
	b_temp = new_vec(n);
	mat_cpy(A, A_temp, n, n);
	vec_cpy(b, b_temp, n);

	// Forward sweep
	for(j=0; j<n; j++){
		// Find the best pivot.
		pivot = j;
		temp = A[j][j];
		for(i=j+1; i<n; i++){
			if(fabs(A[i][j]) > temp){
				temp = A[i][j];
				pivot = i;
			}
		}
		if(pivot != j){
			for(k=j; k<n; k++){
				temp = A[j][k];
				A[j][k] = A[pivot][k];
				A[pivot][k] = temp;
			}
			temp = b[j];
			b[j] = b[pivot];
			b[pivot] = temp;
		}

		for(i=j+1; i<n; i++){
			temp = A[i][j] / A[j][j];
			for(k=0; k<n; k++){
				A[i][k] -= temp*A[j][k];
			}
			b[i] -= temp*b[j];
		}
	}

	// Backward substitution
	x[n-1] = b[n-1] / A[n-1][n-1];
	for(i=n-2; i>=0; i--){
		temp = 0;
		for(j=i+1; j<n; j++){
			temp += A[i][j] * x[j];
		}
		x[i] = (b[i] - temp) / A[i][i];
	}

	mat_cpy(A_temp, A, n, n);
	vec_cpy(b_temp, b, n);
	free_mat(A_temp, n), free_vec(b_temp);
	return;
}

// Solve the linear system PAx = Pb by the preconditioned conjugate gradient method.
void pcg(double **A, double *b, double **P, double *x, int n){
	double rs0 = 0, rsold = 0, rsnew = 0, rzold = 0, rznew = 0, alpha = 0, beta = 0;
	double *r, *p, *Ap, *z;

	int iter = 0, ITER_MAX = n;
	double tol = pow(0.1, 10);

	double *temp_vec1;
	r = new_vec(n);
	p = new_vec(n);
	Ap = new_vec(n);
	z = new_vec(n);
	temp_vec1 = new_vec(n);

	mat_vec_mult(A, x, temp_vec1, n, n);
	vec_lincom(b, temp_vec1, 1, -1, r, n);
	
	if(P){
		mat_vec_mult(P, r, z, n, n);
	}
	else{
		vec_cpy(r, z, n);
	}
	vec_cpy(z, p, n);
	rsold = vec_innerprod(r, r, n);
	rzold = vec_innerprod(r, z, n);
	rs0 = rsold;

	for (iter = 0; iter < ITER_MAX; iter++)
	{
		mat_vec_mult(A, p, Ap, n, n);
		alpha = rzold / vec_innerprod(p, Ap, n);
		vec_lincom(p, p, alpha, 0, p, n);
		vec_lincom(Ap, Ap, alpha, 0, Ap, n);
		vec_lincom(x, p, 1, 1, x, n);
		vec_lincom(r, Ap, 1, -1, r, n);
		rsnew = vec_innerprod(r, r, n);

		// Check the stop condtiion
		if (rsnew / rs0 < tol) {
			//printf("mat_pcg converged with %d iterations.\n", iter+1);
			break;
		}

		if(P){
			mat_vec_mult(P, r, z, n, n);
		}
		else{
			vec_cpy(r, z, n);
		}
		rznew = vec_innerprod(r, z, n);
		beta = rznew / rzold;
		vec_lincom(p, p, beta / alpha, 0, p, n);
		vec_lincom(z, p, 1, 1, p, n);
		rsold = rsnew;
		rzold = rznew;
	}
	if(iter == ITER_MAX){
		//printf("Warning: PCG iterated MAX_ITER iterations but did not converge.\n");
	}

	free_vec(r), free_vec(z), free_vec(p), free_vec(Ap), free_vec(temp_vec1);
	return;
}

// Solve the tridiagonal linear system Ax = b by the Thomas' algorithm.
// A[n][3]
void mat_trisolve(double **A, double *b, double *x, int n){
	int  i = 0;
	double *c, *d;

	c = new_vec(n-1);
	d = new_vec(n);

	// Forward sweep
	c[0] = A[0][2] / A[0][1];
	for(i=1; i<n-1; i++){
		c[i] = A[i][2] / (A[i][1] - A[i][0]*c[i-1]);
	}
	d[0] = b[0] / A[0][1];
	for(i=1; i<n; i++){
		d[i] = (b[i] - A[i][0]*d[i-1]) / (A[i][1] - A[i][0]*c[i-1]);
	}

	// Back substitution
	x[n-1] = d[n-1];
	for(i=n-2; i>=0; i--){
		x[i] = d[i] - c[i]*x[i+1];
	}

	free_vec(c), free_vec(d);
	return;
}

/* ------------------------------------------------------------
Tensor operations
-------------------------------------------------------------*/
// Create a new tensor of size l X m X n
double ***new_ten(int l, int m, int n)
{
	double ***A;
	int i = 0, j = 0, k = 0;

	A = (double ***) malloc(l * sizeof(double **));
	for(i=0; i<l; i++){
		A[i] = new_mat(m, n);
	}

	return A;
}

// Free the l X m X n tensor A
void free_ten(double ***A, int l, int m)
{
	int i = 0, j = 0;
	for (i = 0; i < l; i++)
	{
		free_mat(A[i], m);
	}
	free(A);
	return;
}
