// shallow_nn structure
typedef struct _shallow_nn_
{
    // d: Input dimension
    // N: Number of sampling points for numerical integration
    // n: Number of neurons
    int d, N, n;

    // problem_type: 0 - L^2 function approximation, 1 - Neumann BVP
    int problem_type;

    // x[N][d]: Coodinates of sampling points
    // f[N]: Function f
    // u[N]: Function u
    double **x, *f, *u;

    // double phi(double): Activation function
    // double dphi(double): Derivative of actiation function
    double (*phi)(double); 
    double (*dphi)(double);

} shallow_nn;

// New shallow_nn structure
shallow_nn *new_shallow_nn(int d, int N, int n, int problem_type, double (*fun_f)(double *), double (*fun_u)(double *), double (*phi)(double), double (*dphi)(double));
// Free shallow_nn structure
void free_shallow_nn(shallow_nn *nn);
// He initialization for the parameter c
double *new_c_He(shallow_nn *nn);
// He initialization for theta
double *new_theta_He(shallow_nn *nn);

// NN function defined by the parameters in the shallow_nn structure
void nn_fun(shallow_nn *nn, double *c, double *theta, double *u_nn, double **dudc, double **dudtheta, double **grad_u_nn, double ***grad_dudc, double ***grad_dudtheta);
// Loss function
double loss_fun(shallow_nn *nn, double *c, double *theta, int nograd, double *dEda, double *dEdtheta);

// Train the neural network by gradient descent
double train_GD(shallow_nn *nn, double *c, double *theta, int num_epochs, double *E_decay);
// Train the neural network by Adam
double train_Adam(shallow_nn *nn, double *c, double *theta, int num_epochs, double *E_decay);
// Train the neural network by least squares/gradient descent
double train_LSGD(shallow_nn *nn, double *c, double *theta, int num_epochs, double *E_decay);

// Multiplication by theta = (W, b): theta * x1 = W * x + b
void theta_mult(shallow_nn *nn, double *theta, double **thetax);
// Find the optimal parameter c when theta is given.
void c_minimize(shallow_nn *nn, double *c, double *theta);
// Halton sequence of dimension d and length N
void halton_seq(int d, int N, double **x);
// Normal random number generation by the Marsaglia polar method
// Inputs - mu: mean, sigma: standard deviation
double normrnd(double mu, double sigma);
// L^2-inner product of two functions f and g defined on the grid x
double L2_innerprod(double *f, double *g, shallow_nn *nn);
// a-inner product of two functions f and g defined on the grid x
double a_innerprod(double *f, double **grad_f, double *g, double **grad_g, shallow_nn *nn);
