#!/bin/bash

# 配置
LOG_DIR="inference_logs"  # 日志文件目录
PYTHON_SCRIPT="detect_chunk_new2.py"  # 统一使用的Python脚本

mkdir -p $LOG_DIR

# 为每个 GPU 定义任务列表
# 每个 GPU 的任务列表是一个数组，每个任务是一个参数字符串
# 格式: "input_dir gamma delta beam_num chunk_size mode threshold test_min_tokens"

# GPU 0 的任务列表
GPU0_TASKS=(
    "pred/llama2-7b-chat-4k_no_g0.65_d12.5_all 0.25 5 4 20 old 0.01 6"
)

# GPU 1 的任务列表
GPU1_TASKS=(
    "pred/llama2-7b-chat-4k_no_g0.65_d12.5_all 0.15 10 4 20 old 0.01 6"
)

# GPU 2 的任务列表
GPU2_TASKS=(
    "pred/llama2-7b-chat-4k_no_g0.65_d12.5_all 0.2 9 4 20 gpt 0.01 6"
)

# GPU 3 的任务列表
GPU3_TASKS=(
    "pred/llama2-7b-chat-4k_no_g0.65_d12.5_all 0.3 15 4 20 v2 0.01 6"
)

# GPU 4 的任务列表
GPU4_TASKS=(
    "pred/qwen2.5-7b_no_g0.25_d5.0_alpha0.75_hard_all 0.45 5 4 20 old 0.01 6"
)

# GPU 5 的任务列表
GPU5_TASKS=(
    "pred/qwen2.5-7b_no_g0.25_d5.0_alpha0.75_hard_all 0.05 10 4 20 old 0.01 6"
)

# GPU 6 的任务列表
GPU6_TASKS=(
    "pred/qwen2.5-7b_no_g0.25_d5.0_alpha0.75_hard_all 0.1 9 4 20 gpt 0.01 6"
)

# GPU 7 的任务列表
GPU7_TASKS=(
    "pred/qwen2.5-7b_no_g0.25_d5.0_alpha0.75_hard_all 0.25 11 4 20 v2 0.01 6"
)

# 函数：在指定 GPU 上执行任务列表
execute_gpu_tasks() {
    local gpu_id=$1
    shift 1
    local tasks=("$@")
    
    echo "[$(date)] Starting tasks on GPU $gpu_id using script: $PYTHON_SCRIPT"
    
    for task in "${tasks[@]}"; do
        # 解析任务参数
        IFS=' ' read -r -a params <<< "$task"
        local input_dir="${params[0]}"
        local gamma="${params[1]}"
        local delta="${params[2]}"
        local beam_num="${params[3]}"
        local chunk_size="${params[4]}"
        local mode="${params[5]}"
        local threshold="${params[6]}"
        local test_min_tokens="${params[7]}"
        
        # 生成任务ID
        local task_id="${mode}_${gamma}_${delta}_${beam_num}_${chunk_size}_${threshold}_${test_min_tokens}"
        
        echo "[$(date)] GPU $gpu_id: Starting task $task_id"
        
        # 执行任务
        CUDA_VISIBLE_DEVICES=$gpu_id python $PYTHON_SCRIPT \
            --input_dir "$input_dir" \
            --gamma $gamma \
            --delta $delta \
            --beam_num $beam_num \
            --chunk_size $chunk_size \
            --mode $mode \
            --threshold $threshold \
            --test_min_tokens $test_min_tokens \
            > "${LOG_DIR}/gpu${gpu_id}_${task_id}.log" 2>&1
        
        local exit_code=$?
        if [ $exit_code -eq 0 ]; then
            echo "[$(date)] GPU $gpu_id: ✓ Completed task $task_id"
        else
            echo "[$(date)] GPU $gpu_id: ✗ Failed task $task_id (exit code: $exit_code)"
        fi
    done
    
    echo "[$(date)] All tasks completed on GPU $gpu_id"
}

# 导出函数，以便在子shell中使用
export -f execute_gpu_tasks
export LOG_DIR
export PYTHON_SCRIPT

# 为每个 GPU 启动任务执行进程
if [ ${#GPU0_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 0 "${GPU0_TASKS[@]}" &
    gpu0_pid=$!
fi

if [ ${#GPU1_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 1 "${GPU1_TASKS[@]}" &
    gpu1_pid=$!
fi

if [ ${#GPU2_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 2 "${GPU2_TASKS[@]}" &
    gpu2_pid=$!
fi

if [ ${#GPU3_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 3 "${GPU3_TASKS[@]}" &
    gpu3_pid=$!
fi

if [ ${#GPU4_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 4 "${GPU4_TASKS[@]}" &
    gpu4_pid=$!
fi

if [ ${#GPU5_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 5 "${GPU5_TASKS[@]}" &
    gpu5_pid=$!
fi

if [ ${#GPU6_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 6 "${GPU6_TASKS[@]}" &
    gpu6_pid=$!
fi

if [ ${#GPU7_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 7 "${GPU7_TASKS[@]}" &
    gpu7_pid=$!
fi

# 等待所有 GPU 完成任务
echo "[$(date)] Waiting for all GPUs to complete their tasks..."

if [ -n "$gpu0_pid" ]; then wait $gpu0_pid; fi
if [ -n "$gpu1_pid" ]; then wait $gpu1_pid; fi
if [ -n "$gpu2_pid" ]; then wait $gpu2_pid; fi
if [ -n "$gpu3_pid" ]; then wait $gpu3_pid; fi
if [ -n "$gpu4_pid" ]; then wait $gpu4_pid; fi
if [ -n "$gpu5_pid" ]; then wait $gpu5_pid; fi
if [ -n "$gpu6_pid" ]; then wait $gpu6_pid; fi
if [ -n "$gpu7_pid" ]; then wait $gpu7_pid; fi

echo "[$(date)] All tasks completed on all GPUs!"

# 显示任务总结
echo ""
echo "=== Task Summary ==="
for gpu in 0 1 2 3 4 5 6 7; do
    tasks_var="GPU${gpu}_TASKS[@]"
    tasks_count=$(eval echo \${#$tasks_var})
    
    if [ $tasks_count -gt 0 ]; then
        echo "GPU $gpu: $tasks_count tasks using $PYTHON_SCRIPT"
    fi
done