#!/bin/bash

# 配置
LOG_DIR="inference_logs"  # 日志文件目录
EMBEDDER_MODEL_PATH="/path/to/your/embedder/model"  # 添加embedder模型路径配置

# 为每个 GPU 指定不同的 Python 脚本
GPU0_SCRIPT="pred2_semsim_chunk.py"
GPU1_SCRIPT="pred2_semsim_chunk.py"
GPU2_SCRIPT="pred2_semsim_chunk.py"
GPU3_SCRIPT="pred2_semsim_chunk.py"
GPU4_SCRIPT="pred2_semsim_chunk.py"
GPU5_SCRIPT="pred2_semsim_chunk.py"
GPU6_SCRIPT="pred2_semsim_chunk.py"
GPU7_SCRIPT="pred2_semsim_chunk.py"

mkdir -p $LOG_DIR

# 为每个 GPU 定义任务列表
# 每个 GPU 的任务列表是一个数组，每个任务是一个参数字符串
# 格式: "mode gamma delta bl_type model dataset custom_text K chunk_size alpha"

#    "old 0.25 5 hard llama2-7b-chat-4k all all 4 20 0.75"
#    "old 0.1 10 soft llama2-7b-chat-4k all all 4 20 0.75"
#    "gpt 0.1 10 soft llama2-7b-chat-4k all all 4 20 0.75"
#    "v2 0.25 15 soft llama2-7b-chat-4k all all 4 20 0.75"

# GPU 0 的任务列表
GPU0_TASKS=(
    "old 0.35 5 hard qwen2.5-7b all all 4 20 0.85"
)

# GPU 1 的任务列表
GPU1_TASKS=(
    "old 0.35 5 hard qwen2.5-7b all all 4 20 0.95"
)

# GPU 2 的任务列表
GPU2_TASKS=(
    "gpt 0.1 9 soft qwen2.5-7b all all 4 20 0.75"
)

# GPU 3 的任务列表
# 减小g值
GPU3_TASKS=(
    "gpt 0.1 8 soft qwen2.5-7b all all 4 20 0.75"
)
# "old 0.25 5 hard qwen2.5-7b all all"
# "old 0.1 10 soft qwen2.5-7b all all"
# "gpt 0.1 10 soft qwen2.5-7b all all"
# "v2 0.25 15 soft qwen2.5-7b all all"

# GPU 4 的任务列表
GPU4_TASKS=(
    "gpt 0.1 7 soft qwen2.5-7b all all 4 20 0.75"
)

# GPU 5 的任务列表
GPU5_TASKS=(
    "gpt 0.25 7 soft internlm-7b-8k all all 4 20 0.75"
)

# GPU 6 的任务列表
GPU6_TASKS=(
    "gpt 0.25 6 soft internlm-7b-8k all all 4 20 0.75"
)

# GPU 7 的任务列表 (使用pred.py，增大g值)
GPU7_TASKS=(
    "gpt 0.2 8 soft llama2-7b-chat-4k all all 4 20 0.85"
)

# 函数：在指定 GPU 上执行任务列表
execute_gpu_tasks() {
    local gpu_id=$1
    local python_script=$2
    shift 2
    local tasks=("$@")
    
    echo "[$(date)] Starting tasks on GPU $gpu_id using script: $python_script"
    
    # 获取脚本名称（不含路径和扩展名）用于日志命名
    local script_name=$(basename "$python_script" .py)
    
    for task in "${tasks[@]}"; do
        # 解析任务参数
        IFS=' ' read -r -a params <<< "$task"
        local mode="${params[0]}"
        local gamma="${params[1]}"
        local delta="${params[2]}"
        local bl_type="${params[3]}"
        local model="${params[4]}"
        local dataset="${params[5]}"
        local custom_text="${params[6]}"
        
        # 根据脚本类型判断是否需要额外参数
        local has_extra_params=false
        if [[ "$python_script" == *"pred2_semsim_chunk.py"* ]]; then
            has_extra_params=true
            local K="${params[7]}"
            local chunk_size="${params[8]}"
            local alpha="${params[9]}"
        fi
        
        # 生成任务ID
        if [ "$has_extra_params" = true ]; then
            local task_id="${mode}_${gamma}_${delta}_${bl_type}_${model}_${dataset}_${alpha}_${custom_text}"
        else
            local task_id="${mode}_${gamma}_${delta}_${bl_type}_${model}_${dataset}_${custom_text}"
        fi
        
        echo "[$(date)] GPU $gpu_id: Starting task $task_id with $python_script"
        
        # 执行任务 - 根据脚本类型传递不同参数
        if [ "$has_extra_params" = true ]; then
            # pred2_semsim_chunk.py - 传递所有参数，包括embedder_model_path
            CUDA_VISIBLE_DEVICES=$gpu_id python $python_script \
                --mode $mode \
                --gamma $gamma \
                --delta $delta \
                --bl_type $bl_type \
                --model $model \
                --dataset "$dataset" \
                --custom_text "$custom_text" \
                --K $K \
                --chunk_size $chunk_size \
                --alpha $alpha \
                --embedder_model_path "$EMBEDDER_MODEL_PATH" \
                > "${LOG_DIR}/gpu${gpu_id}_${script_name}_${task_id}.log" 2>&1
        else
            # pred.py - 只传递基本参数
            CUDA_VISIBLE_DEVICES=$gpu_id python $python_script \
                --mode $mode \
                --gamma $gamma \
                --delta $delta \
                --bl_type $bl_type \
                --model $model \
                --dataset "$dataset" \
                --custom_text "$custom_text" \
                > "${LOG_DIR}/gpu${gpu_id}_${script_name}_${task_id}.log" 2>&1
        fi
        
        local exit_code=$?
        if [ $exit_code -eq 0 ]; then
            echo "[$(date)] GPU $gpu_id: ✓ Completed task $task_id"
        else
            echo "[$(date)] GPU $gpu_id: ✗ Failed task $task_id (exit code: $exit_code)"
        fi
    done
    
    echo "[$(date)] All tasks completed on GPU $gpu_id"
}

# 导出函数和变量，以便在子shell中使用
export -f execute_gpu_tasks
export LOG_DIR
export EMBEDDER_MODEL_PATH

# 为每个 GPU 启动任务执行进程
if [ ${#GPU0_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 0 "$GPU0_SCRIPT" "${GPU0_TASKS[@]}" &
    gpu0_pid=$!
fi

if [ ${#GPU1_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 1 "$GPU1_SCRIPT" "${GPU1_TASKS[@]}" &
    gpu1_pid=$!
fi

if [ ${#GPU2_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 2 "$GPU2_SCRIPT" "${GPU2_TASKS[@]}" &
    gpu2_pid=$!
fi

if [ ${#GPU3_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 3 "$GPU3_SCRIPT" "${GPU3_TASKS[@]}" &
    gpu3_pid=$!
fi

if [ ${#GPU4_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 4 "$GPU4_SCRIPT" "${GPU4_TASKS[@]}" &
    gpu4_pid=$!
fi

if [ ${#GPU5_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 5 "$GPU5_SCRIPT" "${GPU5_TASKS[@]}" &
    gpu5_pid=$!
fi

if [ ${#GPU6_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 6 "$GPU6_SCRIPT" "${GPU6_TASKS[@]}" &
    gpu6_pid=$!
fi

if [ ${#GPU7_TASKS[@]} -gt 0 ]; then
    execute_gpu_tasks 7 "$GPU7_SCRIPT" "${GPU7_TASKS[@]}" &
    gpu7_pid=$!
fi

# 等待所有 GPU 完成任务
echo "[$(date)] Waiting for all GPUs to complete their tasks..."

if [ -n "$gpu0_pid" ]; then wait $gpu0_pid; fi
if [ -n "$gpu1_pid" ]; then wait $gpu1_pid; fi
if [ -n "$gpu2_pid" ]; then wait $gpu2_pid; fi
if [ -n "$gpu3_pid" ]; then wait $gpu3_pid; fi
if [ -n "$gpu4_pid" ]; then wait $gpu4_pid; fi
if [ -n "$gpu5_pid" ]; then wait $gpu5_pid; fi
if [ -n "$gpu6_pid" ]; then wait $gpu6_pid; fi
if [ -n "$gpu7_pid" ]; then wait $gpu7_pid; fi

echo "[$(date)] All tasks completed on all GPUs!"

# 显示任务总结
echo ""
echo "=== Task Summary ==="
for gpu in 0 1 2 3 4 5 6 7; do
    script_var="GPU${gpu}_SCRIPT"
    tasks_var="GPU${gpu}_TASKS[@]"
    script_name=${!script_var}
    tasks_count=$(eval echo \${#$tasks_var})
    
    if [ $tasks_count -gt 0 ]; then
        echo "GPU $gpu: $tasks_count tasks using $script_name"
    fi
done