import bpy
import random
from mathutils import Vector
import os
import math
import sys
import json
import mathutils
from mathutils import Matrix, Vector, Euler
from typing import IO, Union, Dict, List, Set, Optional, Any
import numpy as np
import ssl
import cv2
import imageio.v2 as imageio
from PIL import Image

# pip install
# /opt/blender-3.5.1/3.5/python/bin/python3.10 -m pip install imageio[ffmpeg]


RESOLUTION = 256
ENGINE = "BLENDER_EEVEE" # "CYCLES"
RENDER_SAMPLES = 128
META_FILENAME = "meta.json"
SAVE_VIDEO = True

# Render options
ONLY_RENDER_COLOR = False   # TODO: not implemented yet

# Camera layout
CAMERA_ELEVATIONS = [-10, 0, 10, 20, 30, 40] # [-10, 0, 10, 20, 30, 40]
CAMERA_NUM_PER_LAYER = 16

# Environment light
LIGHT_MODE = "default"  # Options: default, zero123
ENV_LIGHT_INTENSITY = 0.8   # For global environment light
ENV_TEXTURES_PATH = "/app/env_textures"
ENV_MAP_INDEX = None
ENV_TEXTURES_LIST = [
    "belfast_sunset_puresky_1k.exr",
    "brown_photostudio_02_1k.exr",
    "city.exr",
    "clarens_midday_1k.exr",
    "courtyard.exr",
    "evening_road_01_puresky_1k.exr",
    "industrial_sunset_puresky_1k.exr",
    "interior.exr",
    "kloofendal_overcast_puresky_1k.exr",
    "kloppenheim_06_puresky_1k.exr",
    "promenade_de_vidy_1k.exr",
    "resting_place_1k.exr",
    "studio_small_09_1k.exr",
    "sunset.exr",
]


def convert_to_webp(src, dst):
    image = Image.open(src)
    image.save(dst, "webp", quality=100)

def rgba_to_rgb(rgba_image, bg_color=[255, 255, 255]):
    background = np.array(bg_color)

    # 分离 RGBA 通道
    foreground = rgba_image[..., :3]
    alpha = rgba_image[..., 3:]

    # 混合前景和背景
    foreground = foreground.astype(float)
    background = background.astype(float)
    alpha = alpha.astype(float)/255

    # 计算 RGB 值
    rgb_image = alpha * foreground + (1 - alpha) * background
    return rgb_image.astype(np.uint8)


def generate_frames(idx):
    param = [
        {
            "type": "render",
            "name": "{}",
            "height": RESOLUTION,
            "width": RESOLUTION,
        }
    ]
    variables = ["render_" + idx + ".webp"]
    for i in range(len(variables)):
        param[i]["name"] = variables[i]
    return param


def convert_position(location, center):
    position = ""
    axis = ["x", "y", "z"]
    sub = location - center
    for i in range(len(axis)):
        if sub[i] > 0:
            position = position + "+" + axis[i]
        elif sub[i] < 0:
            position = position + "-" + axis[i]
    return position


def load_image(file_path: str, num_channels: int = 3) -> np.ndarray:
    """Load the image at the given path returns its pixels as a numpy array.

    The alpha channel is neglected.

    :param file_path: The path to the image.
    :param num_channels: Number of channels to return.
    :return: The numpy array
    """
    file_ending = file_path[file_path.rfind(".") + 1 :].lower()
    if file_ending in ["exr", "png"]:
        return imageio.imread(file_path)[:, :, :num_channels]
    elif file_ending in ["jpg"]:
        img = cv2.imread(file_path)  # reads an image in the BGR format
        img = cv2.cvtColor(img, cv2.COLOR_BGR2RGB)
        return img
    else:
        raise NotImplementedError(
            "File with ending " + file_ending + " cannot be loaded."
        )


def build_transformation_mat(translation, rotation) -> np.ndarray:
    """Build a transformation matrix from translation and rotation parts.

    :param translation: A (3,) vector representing the translation part.
    :param rotation: A 3x3 rotation matrix or Euler angles of shape (3,).
    :return: The 4x4 transformation matrix.
    """
    translation = np.array(translation)
    rotation = np.array(rotation)

    mat = np.eye(4)
    if translation.shape[0] == 3:
        mat[:3, 3] = translation
    else:
        raise RuntimeError(
            f"Translation has invalid shape: {translation.shape}. Must be (3,) or (3,1) vector."
        )
    if rotation.shape == (3, 3):
        mat[:3, :3] = rotation
    elif rotation.shape[0] == 3:
        mat[:3, :3] = np.array(Euler(rotation).to_matrix())
    else:
        raise RuntimeError(
            f"Rotation has invalid shape: {rotation.shape}. Must be rotation matrix of shape "
            f"(3,3) or Euler angles of shape (3,) or (3,1)."
        )

    return mat


def reset_keyframes() -> None:
    """Removes registered keyframes from all objects and resets frame_start and frame_end"""
    bpy.context.scene.frame_start = 0
    bpy.context.scene.frame_end = 0
    for a in bpy.data.actions:
        bpy.data.actions.remove(a)


def get_local2world_mat(blender_obj) -> np.ndarray:
    """Returns the pose of the object in the form of a local2world matrix.
    :return: The 4x4 local2world matrix.
    """
    obj = blender_obj
    # Start with local2parent matrix (if obj has no parent, that equals local2world)
    matrix_world = obj.matrix_basis

    # Go up the scene graph along all parents
    while obj.parent is not None:
        # Add transformation to parent frame
        matrix_world = (
            obj.parent.matrix_basis @ obj.matrix_parent_inverse @ matrix_world
        )
        obj = obj.parent

    return np.array(matrix_world)


def add_camera_pose(cam2world_matrix) -> int:
    if not isinstance(cam2world_matrix, Matrix):
        cam2world_matrix = Matrix(cam2world_matrix)

    cam_ob = bpy.context.scene.camera
    cam_ob.matrix_world = cam2world_matrix

    frame = bpy.context.scene.frame_end
    if bpy.context.scene.frame_end < frame + 1:
        bpy.context.scene.frame_end = frame + 1

    cam_ob.keyframe_insert(data_path="location", frame=frame)
    cam_ob.keyframe_insert(data_path="rotation_euler", frame=frame)
    return frame


def enable_normals_output(output_dir: Optional[str] = "", file_prefix: str = "normal_"):
    bpy.context.scene.render.use_compositing = True
    bpy.context.scene.use_nodes = True

    tree = bpy.context.scene.node_tree

    if "Render Layers" not in tree.nodes:
        rl = tree.nodes.new("CompositorNodeRLayers")
    else:
        rl = tree.nodes["Render Layers"]
    bpy.context.view_layer.use_pass_normal = True
    bpy.context.scene.render.use_compositing = True

    separate_rgba = tree.nodes.new("CompositorNodeSepRGBA")
    space_between_nodes_x = 200
    space_between_nodes_y = -300
    separate_rgba.location.x = space_between_nodes_x
    separate_rgba.location.y = space_between_nodes_y
    tree.links.new(rl.outputs["Normal"], separate_rgba.inputs["Image"])

    combine_rgba = tree.nodes.new("CompositorNodeCombRGBA")
    combine_rgba.location.x = space_between_nodes_x * 14

    c_channels = ["R", "G", "B"]
    offset = space_between_nodes_x * 2
    multiplication_values: List[List[bpy.types.Node]] = [[], [], []]
    channel_results = {}
    for row_index, channel in enumerate(c_channels):
        # matrix multiplication
        mulitpliers = []
        for column in range(3):
            multiply = tree.nodes.new("CompositorNodeMath")
            multiply.operation = "MULTIPLY"
            # setting at the end for all frames
            multiply.inputs[1].default_value = 0
            multiply.location.x = column * space_between_nodes_x + offset
            multiply.location.y = row_index * space_between_nodes_y
            tree.links.new(
                separate_rgba.outputs[c_channels[column]], multiply.inputs[0]
            )
            mulitpliers.append(multiply)
            multiplication_values[row_index].append(multiply)

        first_add = tree.nodes.new("CompositorNodeMath")
        first_add.operation = "ADD"
        first_add.location.x = space_between_nodes_x * 5 + offset
        first_add.location.y = row_index * space_between_nodes_y
        tree.links.new(mulitpliers[0].outputs["Value"], first_add.inputs[0])
        tree.links.new(mulitpliers[1].outputs["Value"], first_add.inputs[1])

        second_add = tree.nodes.new("CompositorNodeMath")
        second_add.operation = "ADD"
        second_add.location.x = space_between_nodes_x * 6 + offset
        second_add.location.y = row_index * space_between_nodes_y
        tree.links.new(first_add.outputs["Value"], second_add.inputs[0])
        tree.links.new(mulitpliers[2].outputs["Value"], second_add.inputs[1])

        channel_results[channel] = second_add

    rot_around_x_axis = mathutils.Matrix.Rotation(math.radians(-90.0), 4, "X")
    for frame in range(bpy.context.scene.frame_start, bpy.context.scene.frame_end):
        bpy.context.scene.frame_set(frame)
        used_rotation_matrix = (
            get_local2world_mat(bpy.context.scene.camera) @ rot_around_x_axis
        )
        for row_index in range(3):
            for column_index in range(3):
                current_multiply = multiplication_values[row_index][column_index]
                current_multiply.inputs[1].default_value = used_rotation_matrix[
                    column_index
                ][row_index]
                current_multiply.inputs[1].keyframe_insert(
                    data_path="default_value", frame=frame
                )

    offset = 8 * space_between_nodes_x
    for index, channel in enumerate(c_channels):
        multiply = tree.nodes.new("CompositorNodeMath")
        multiply.operation = "MULTIPLY"
        multiply.location.x = space_between_nodes_x * 2 + offset
        multiply.location.y = index * space_between_nodes_y
        tree.links.new(channel_results[channel].outputs["Value"], multiply.inputs[0])
        multiply.inputs[1].default_value = 0.5
        if channel == "G":
            multiply.inputs[1].default_value = -0.5
        add = tree.nodes.new("CompositorNodeMath")
        add.operation = "ADD"
        add.location.x = space_between_nodes_x * 3 + offset
        add.location.y = index * space_between_nodes_y
        tree.links.new(multiply.outputs["Value"], add.inputs[0])
        add.inputs[1].default_value = 0.5
        output_channel = channel
        if channel == "G":
            output_channel = "B"
        elif channel == "B":
            output_channel = "G"
        tree.links.new(add.outputs["Value"], combine_rgba.inputs[output_channel])

    normal_file_output = tree.nodes.new("CompositorNodeOutputFile")
    normal_file_output.base_path = output_dir
    normal_file_output.format.file_format = "OPEN_EXR"
    normal_file_output.format.color_mode = "RGBA"
    normal_file_output.format.color_depth = "32"
    normal_file_output.location.x = space_between_nodes_x * 15
    normal_file_output.file_slots.values()[0].path = file_prefix
    tree.links.new(combine_rgba.outputs["Image"], normal_file_output.inputs["Image"])


def enable_depth_output(output_dir: Optional[str] = "", file_prefix: str = "depth_"):
    bpy.context.scene.render.use_compositing = True
    bpy.context.scene.use_nodes = True

    tree = bpy.context.scene.node_tree
    links = tree.links

    if "Render Layers" not in tree.nodes:
        rl = tree.nodes.new("CompositorNodeRLayers")
    else:
        rl = tree.nodes["Render Layers"]
    bpy.context.view_layer.use_pass_z = True

    depth_output = tree.nodes.new("CompositorNodeOutputFile")
    depth_output.base_path = output_dir
    depth_output.name = "DepthOutput"
    depth_output.format.file_format = "OPEN_EXR"
    depth_output.format.color_depth = "32"
    depth_output.file_slots.values()[0].path = file_prefix

    links.new(rl.outputs["Depth"], depth_output.inputs["Image"])


def enable_albedo_output(output_dir: Optional[str] = "", file_prefix: str = "albedo_"):
    bpy.context.scene.render.use_compositing = True
    bpy.context.scene.use_nodes = True

    tree = bpy.context.scene.node_tree

    if "Render Layers" not in tree.nodes:
        rl = tree.nodes.new("CompositorNodeRLayers")
    else:
        rl = tree.nodes["Render Layers"]
    bpy.context.view_layer.use_pass_diffuse_color = True

    alpha_albedo = tree.nodes.new(type="CompositorNodeSetAlpha")
    tree.links.new(rl.outputs["DiffCol"], alpha_albedo.inputs["Image"])
    tree.links.new(rl.outputs["Alpha"], alpha_albedo.inputs["Alpha"])

    albedo_file_output = tree.nodes.new(type="CompositorNodeOutputFile")
    albedo_file_output.base_path = output_dir
    albedo_file_output.file_slots[0].use_node_format = True
    albedo_file_output.format.file_format = "PNG"
    albedo_file_output.format.color_mode = "RGBA"
    albedo_file_output.format.color_depth = "16"
    albedo_file_output.file_slots.values()[0].path = file_prefix

    tree.links.new(alpha_albedo.outputs["Image"], albedo_file_output.inputs[0])


def render():
    # bpy.context.scene.render.use_compositing = True
    bpy.context.scene.use_nodes = True

    tree = bpy.context.scene.node_tree
    links = tree.links

    if "Render Layers" not in tree.nodes:
        rl = tree.nodes.new("CompositorNodeRLayers")
    else:
        rl = tree.nodes["Render Layers"]
    if bpy.context.scene.frame_end != bpy.context.scene.frame_start:
        bpy.context.scene.frame_end -= 1
        bpy.ops.render.render(animation=True, write_still=True)
        bpy.context.scene.frame_end += 1
    else:
        raise RuntimeError(
            "No camera poses have been registered, therefore nothing can be rendered. A camera "
            "pose can be registered via bproc.camera.add_camera_pose()."
        )


def set_color_output(output_dir: Optional[str] = "", file_prefix: str = "render_"):
    scene = bpy.context.scene
    scene.render.resolution_x = RESOLUTION
    scene.render.resolution_y = RESOLUTION
    scene.render.resolution_percentage = 100
    scene.render.image_settings.file_format = "PNG"
    scene.render.image_settings.color_mode = "RGBA"
    scene.render.image_settings.color_depth = "16"
    scene.render.film_transparent = True
    scene.render.filepath = os.path.join(output_dir, file_prefix)
    pass


def eevee_init():
    bpy.context.scene.render.engine = "BLENDER_EEVEE"
    bpy.context.scene.eevee.taa_render_samples = RENDER_SAMPLES
    bpy.context.scene.eevee.use_gtao = True
    bpy.context.scene.eevee.use_ssr = True
    # bpy.context.scene.eevee.use_bloom = True
    bpy.context.scene.render.use_high_quality_normals = True


def cycles_init():
    bpy.context.scene.render.engine = "CYCLES"
    bpy.context.scene.cycles.samples = RENDER_SAMPLES
    bpy.context.scene.cycles.diffuse_bounces = 1
    bpy.context.scene.cycles.glossy_bounces = 1
    bpy.context.scene.cycles.transparent_max_bounces = 3
    bpy.context.scene.cycles.transmission_bounces = 3
    bpy.context.scene.cycles.filter_width = 0.01
    bpy.context.scene.cycles.use_denoising = True
    bpy.context.scene.render.film_transparent = True


def set_global_light(env_light=0.5):
    world_tree = bpy.context.scene.world.node_tree
    back_node = world_tree.nodes["Background"]
    back_node.inputs["Color"].default_value = Vector(
        [env_light, env_light, env_light, 1.0]
    )
    back_node.inputs["Strength"].default_value = 1.0


def set_environment_map(environment_name):
    world = bpy.context.scene.world
    if not world:
        world = bpy.data.worlds.new("World")
        bpy.context.scene.world = world
    else:
        world.use_nodes = True
        world.node_tree.nodes.clear()

    env_texture_node = world.node_tree.nodes.new(type="ShaderNodeTexEnvironment")
    env_texture_node.location = (0, 0)

    bg_node = world.node_tree.nodes.new(type="ShaderNodeBackground")
    bg_node.location = (400, 0)

    output_node = world.node_tree.nodes.new(type="ShaderNodeOutputWorld")
    output_node.location = (800, 0)

    links = world.node_tree.links
    links.new(env_texture_node.outputs["Color"], bg_node.inputs["Color"])
    links.new(bg_node.outputs["Background"], output_node.inputs["Surface"])

    bpy.ops.image.open(filepath=os.path.join(ENV_TEXTURES_PATH, environment_name))
    env_texture_node.image = bpy.data.images.get(environment_name)


def clear_scene():
    # SyncDreamer
    if LIGHT_MODE == "zero123":
        for obj in bpy.data.objects:
            if obj.type not in {"CAMERA", "LIGHT"}:
                bpy.data.objects.remove(obj, do_unlink=True)
    else:
        bpy.ops.object.select_all(action="SELECT")
        bpy.ops.object.delete()
    bpy.context.scene.use_nodes = True
    node_tree = bpy.context.scene.node_tree

    # 清空节点树中的所有节点
    for node in node_tree.nodes:
        node_tree.nodes.remove(node)
    reset_keyframes()


def import_models(filepath, types):
    if types == "glb":
        bpy.ops.import_scene.gltf(filepath=filepath)
    elif types == "ply":
        bpy.ops.import_mesh.ply(filepath=filepath)
    elif types == "obj_yz":
        if filepath.endswith(".obj"):
            obj_path = filepath
            dirpath = os.path.dirname(filepath)
        else:
            obj_path = os.path.join(filepath, "model.obj")
            dirpath = filepath
        base_name = os.path.basename(obj_path)
        bpy.ops.wm.obj_import(
            filepath=obj_path,
            directory=dirpath,
            files=[{"name": base_name}],
            forward_axis='Y',
            up_axis='Z',
        )


def scene_bbox(single_obj=None, ignore_matrix=False):
    bbox_min = (math.inf,) * 3
    bbox_max = (-math.inf,) * 3
    found = False
    for obj in scene_meshes() if single_obj is None else [single_obj]:
        found = True
        for coord in obj.bound_box:
            coord = Vector(coord)
            if not ignore_matrix:
                coord = obj.matrix_world @ coord
            bbox_min = tuple(min(x, y) for x, y in zip(bbox_min, coord))
            bbox_max = tuple(max(x, y) for x, y in zip(bbox_max, coord))
    if not found:
        raise RuntimeError("no objects in scene to compute bounding box for")
    return Vector(bbox_min), Vector(bbox_max)


def scene_root_objects():
    for obj in bpy.context.scene.objects.values():
        if not obj.parent:
            yield obj


def scene_meshes():
    for obj in bpy.context.scene.objects.values():
        if isinstance(obj.data, (bpy.types.Mesh)):
            yield obj


def normalize_scene(normalization_range):
    bbox_min, bbox_max = scene_bbox()
    scale = normalization_range / max(bbox_max - bbox_min)
    for obj in scene_root_objects():
        obj.scale = obj.scale * scale
    # Apply scale to matrix_world.
    bpy.context.view_layer.update()
    bbox_min, bbox_max = scene_bbox()
    offset = -(bbox_min + bbox_max) / 2
    for obj in scene_root_objects():
        obj.matrix_world.translation += offset
    bpy.ops.object.select_all(action="DESELECT")


def compute_bounding_box(mesh_objects):
    min_coords = Vector((float("inf"), float("inf"), float("inf")))
    max_coords = Vector((float("-inf"), float("-inf"), float("-inf")))

    for obj in mesh_objects:
        matrix_world = obj.matrix_world
        mesh = obj.data

        for vert in mesh.vertices:
            global_coord = matrix_world @ vert.co

            min_coords = Vector((min(min_coords[i], global_coord[i]) for i in range(3)))
            max_coords = Vector((max(max_coords[i], global_coord[i]) for i in range(3)))

    bbox_center = (min_coords + max_coords) / 2
    bbox_size = max_coords - min_coords

    return bbox_center, bbox_size


def get_camera_positions_on_sphere(center, radius):
    points = []
    elevation_t = []
    azimuth_t = []

    elevation_deg = CAMERA_ELEVATIONS
    elevation = np.deg2rad(elevation_deg)
    azimuth_deg = np.linspace(0 + CAMERA_AZIMUTH_BIAS, 360 + CAMERA_AZIMUTH_BIAS, CAMERA_NUM_PER_LAYER + 1)[:-1]
    azimuth = np.deg2rad(azimuth_deg)

    for theta in azimuth:
        for _phi in elevation:
            phi = 0.5 * math.pi - _phi
            elevation_t.append(_phi)
            azimuth_t.append(theta)
            
            r = radius
            x = center[0] + r * math.sin(phi) * math.cos(theta)
            y = center[1] + r * math.sin(phi) * math.sin(theta)
            z = center[2] + r * math.cos(phi)
            points.append(Vector((x, y, z)))

    return points, elevation_t, azimuth_t


def listify_matrix(matrix):
    matrix_list = []
    for row in matrix:
        matrix_list.append(list(row))
    return matrix_list


def enable_pbr_output(output_dir, attr_name, color_mode="RGBA", file_prefix: str = ""):
    if file_prefix == "":
        file_prefix = attr_name.lower().replace(" ", "-") + "_"

    for material in bpy.data.materials:
        material.use_nodes = True
        node_tree = material.node_tree
        nodes = node_tree.nodes
        roughness_input = nodes["Principled BSDF"].inputs[attr_name]
        if roughness_input.is_linked:
            linked_node = roughness_input.links[0].from_node
            linked_socket = roughness_input.links[0].from_socket

            aov_output = nodes.new("ShaderNodeOutputAOV")
            aov_output.name = attr_name
            node_tree.links.new(linked_socket, aov_output.inputs[0])

        else:
            fixed_roughness = roughness_input.default_value
            if isinstance(fixed_roughness, float):
                roughness_value = nodes.new("ShaderNodeValue")
                input_idx = 1
            else:
                roughness_value = nodes.new("ShaderNodeRGB")
                input_idx = 0

            roughness_value.outputs[0].default_value = fixed_roughness

            aov_output = nodes.new("ShaderNodeOutputAOV")
            aov_output.name = attr_name
            node_tree.links.new(roughness_value.outputs[0], aov_output.inputs[0])

    tree = bpy.context.scene.node_tree
    links = tree.links
    if "Render Layers" not in tree.nodes:
        rl = tree.nodes.new("CompositorNodeRLayers")
    else:
        rl = tree.nodes["Render Layers"]

    roughness_file_output = tree.nodes.new(type="CompositorNodeOutputFile")
    roughness_file_output.base_path = output_dir
    roughness_file_output.file_slots[0].use_node_format = True
    roughness_file_output.format.file_format = "PNG"
    roughness_file_output.format.color_mode = color_mode
    roughness_file_output.format.color_depth = "16"
    roughness_file_output.file_slots.values()[0].path = file_prefix

    bpy.ops.scene.view_layer_add_aov()
    bpy.context.scene.view_layers["ViewLayer"].active_aov.name = attr_name
    roughness_alpha = tree.nodes.new(type="CompositorNodeSetAlpha")
    tree.links.new(rl.outputs[attr_name], roughness_alpha.inputs["Image"])
    tree.links.new(rl.outputs["Alpha"], roughness_alpha.inputs["Alpha"])

    links.new(roughness_alpha.outputs["Image"], roughness_file_output.inputs["Image"])


def add_cameras(location_list, camera_empty):
    camera_list = []
    for item in location_list:
        bpy.ops.object.camera_add(location=item)
        _camera = bpy.context.selected_objects[0]
        camera_list.append(_camera)
        _constraint = _camera.constraints.new(type="TRACK_TO")
        _constraint.track_axis = "TRACK_NEGATIVE_Z"
        _constraint.up_axis = "UP_Y"
        _camera.parent = camera_empty
        _constraint.target = camera_empty
        _constraint.owner_space = "LOCAL"
    return camera_list


def process(filepath, types, output_path):
    if ENGINE == "CYCLES":
        cycles_init()
    elif ENGINE == "BLENDER_EEVEE":
        eevee_init()

    clear_scene()
    import_models(filepath, types)

    bpy.ops.object.select_by_type(type="MESH")
    os.makedirs(output_path, exist_ok=True)

    mesh_objects = [obj for obj in bpy.context.scene.objects if obj.type == "MESH"]
    print("mesh_objects: {}".format(len(mesh_objects)))

    for obj in mesh_objects:
        obj.data.use_auto_smooth = True
        obj.data.auto_smooth_angle = np.deg2rad(30)

    for obj in bpy.data.objects:
        if obj.animation_data is not None:
            obj.animation_data_clear()

    normalization_range = 1
    normalize_scene(normalization_range)
    bbox_center, bbox_size = compute_bounding_box(mesh_objects)
    bbox_min, bbox_max = scene_bbox()
    bbox_center = Vector((0, 0, 0))
    bbox_size = Vector(bbox_max) - Vector(bbox_min)

    # ratio = 1.0
    # _tmp = math.sqrt(bbox_size.x**2 + bbox_size.y**2 + bbox_size.z**2)

    # distance = max(bbox_size.x, bbox_size.y, bbox_size.z) * ratio / 0.72 * _tmp
    distance = 1.5

    print("distance is {}".format(distance))

    bpy.ops.object.camera_add(location=(0, 0, 0))
    camera = bpy.context.object
    camera_lens = 35
    camera_sensor_width = 32
    # camera.data.angle = math.radians(70)
    camera.data.lens = camera_lens
    camera.data.sensor_width = camera_sensor_width
    bpy.context.scene.camera = camera

    idx = 0

    env_texture = "null"
    if ENV_MAP_INDEX is not None:
        if ENV_MAP_INDEX == -1:
            env_texture = random.choice(ENV_TEXTURES_LIST)
        else:
            env_texture = ENV_TEXTURES_LIST[ENV_MAP_INDEX]
        set_environment_map(env_texture)
    else:
        set_global_light(env_light=ENV_LIGHT_INTENSITY)

    all_camera_locations, ele, azi = get_camera_positions_on_sphere(
        bbox_center, distance
    )

    camera_angle_x = 2.0 * math.atan(camera_sensor_width / 2 / camera.data.lens)
    out_data = {
        "distance": distance,
        "camera_angle_x": camera_angle_x,
        "sensor_width": camera_sensor_width,
        "env_texture": env_texture,
        "locations": [],
    }

    for camera_location in all_camera_locations:
        rotation_euler = (
            (bbox_center - camera_location).to_track_quat("-Z", "Y").to_euler()
        )
        cam_matrix = build_transformation_mat(camera_location, rotation_euler)
        add_camera_pose(cam_matrix)
        index = "{0:04d}".format(idx)
        out_data["locations"].append(
            {
                "index": index,
                "transform_matrix": listify_matrix(cam_matrix),
                "elevation": ele[idx],
                "azimuth": azi[idx],
                "frames": generate_frames(index),
            }
        )
        idx += 1

    set_color_output(output_dir=output_path)
    # enable_normals_output(output_dir=output_path)
    # enable_depth_output(output_dir=output_path)
    # enable_albedo_output(output_dir=output_path)
    # enable_pbr_output(output_dir=output_path,
    #                   attr_name="Roughness", color_mode='RGBA')
    # enable_pbr_output(output_dir=output_path,
    #                   attr_name="Base Color", color_mode='RGBA')
    # enable_pbr_output(output_dir=output_path,
    #                   attr_name="Metallic", color_mode='RGBA')

    render()
    with open(os.path.join(output_path, META_FILENAME), "w") as out_file:
        json.dump(out_data, out_file, indent=4)

    # for i in range(idx):
    #     index = "{0:04d}".format(i)
    #     filepath = os.path.join(output_path, f"normal_{index}.exr")
    #     data = load_image(filepath, 3)
    #     normal_map = data[:, :, ::-1]*255
    #     cv2.imwrite(os.path.join(
    #         output_path, 'normal_{}.png'.format(index)), normal_map)
    #     os.remove(os.path.join(output_path, f"normal_{index}.exr"))
    
    color_images = []
    for i in range(idx):
        index = "{0:04d}".format(i)
        filepath = os.path.join(output_path, f"render_{index}.png")
        color_images.append(imageio.imread(filepath))
    
    # Save video
    if SAVE_VIDEO:
        rgb_images = [rgba_to_rgb(img) for img in color_images]
        imageio.mimsave(os.path.join(output_path, "render.mp4"), rgb_images, fps=24)

    for i in range(idx):
        index = "{0:04d}".format(i)
        for i in ["render_"]:
            filepath = os.path.join(output_path, f"{i}{index}.png")
            convert_to_webp(filepath, filepath.replace(".png", ".webp"))
            os.remove(filepath)


# process(
#     "/mnt/pfs/users/huangzehuan/project/instant-nsr-pl/exp/neus-gt/GSO_0000@20231113-124939/save/it3000-mc512.obj",
#     "obj_yz",
#     "output/GSO/GSO_0000_0",
# )

CAMERA_AZIMUTH_BIAS = 270
CAMERA_ELEVATIONS = [-10, 0, 10, 20, 30, 40]
RESOLUTION = 256

process(
    "more_models/dog_toy.glb",
    "glb",
    "more_data/dog_toy",
)

# if __name__ == "__main__":
#     if len(sys.argv) < 1:
#         print(
#             "Usage: [path_to_blender] -b -P render.py [dir] [types] [uid] [output_path]"
#         )
#         exit(1)
#     else:
#         root_dir = sys.argv[4]
#         types = sys.argv[5]
#         output_path = sys.argv[6]
#         ret = process(root_dir, types, output_path)
#         if ret:
#             exit(0)
#  
